package com.namasoft.common.utils.translation;

import com.namasoft.common.constants.Language;
import com.namasoft.common.utilities.*;
import org.apache.commons.io.IOUtils;

import java.io.*;
import java.text.MessageFormat;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;

public class MessageTranslatorFromFile extends MessageTranslatorBase
{
	private static final String LINE_COMMENT = "#";
	private List<String> keys = new ArrayList<>();
	private Map<String, String> arabicMap;
	private Map<String, String> englishMap;
	private final String englishFile;
	private final String arabicFile;
	boolean classPath;
	private long order = 0;

	public static MessageTranslatorFromFile fromClassPathFiles(String arabicFile, String englishFile, long order)
	{
		return new MessageTranslatorFromFile(arabicFile, englishFile, true, order);
	}

	public static MessageTranslatorFromFile fromPhysicalFiles(String arabicFile, String englishFile, long order)
	{
		return new MessageTranslatorFromFile(arabicFile, englishFile, false, order);
	}

	public MessageTranslatorFromFile(String arabicFile, String englishFile, boolean classPath, long order)
	{
		this.englishFile = englishFile;
		this.arabicFile = arabicFile;
		this.classPath = classPath;
		this.order = order;
		loadMaps();
	}

	private void loadMaps()
	{
		if (arabicFile != null)
			arabicMap = loadMap(arabicFile);
		if (englishFile != null)
			englishMap = loadMap(englishFile);
	}

	public enum FixingWhat
	{
		Empty
				{
					@Override
					void empty(int i)
					{
						System.out.println(i);
					}
				}, Equal
			{
				@Override
				void equally(int i)
				{
					System.out.println(i);
				}
			}, NotEqual
			{
				@Override
				void notEqually(int i)
				{
					System.out.println(i);
				}

			}, All
			{
				@Override
				void equally(int i)
				{
					System.out.println(i);
				}

				@Override
				void empty(int i)
				{
					System.out.println(i);
				}

				@Override
				void notEqually(int i)
				{
					System.out.println(i);
				}
			}, None;

		void empty(int i)
		{

		}

		void equally(int i)
		{

		}

		void notEqually(int i)
		{

		}

	}

	public static FixingWhat fixingWhat = FixingWhat.Empty;
	public static BiConsumer<Integer, String> spacesReceiver = null;
	public Map<String, String> loadMap(String file)
	{
		InputStream resourceAsStream = null;
		try
		{
			resourceAsStream = classPath ? MessageTranslatorFromFile.class.getClassLoader().getResourceAsStream(file) : new FileInputStream(file);
		}
		catch (FileNotFoundException e)
		{
			NaMaLogger.error(e);
			return new ConcurrentHashMap<>();
		}
		if (resourceAsStream == null)
			return new ConcurrentHashMap<>();
		ConcurrentHashMap<String, String> hashMap = new ConcurrentHashMap<>();
		keys = new ArrayList<>();
		Scanner scanner = new Scanner(resourceAsStream, "utf-8");
		int i = 0;
		Map<String, Integer> oldIndices = new ConcurrentHashMap<>();
		while (scanner.hasNextLine())
		{
			i++;
			String line = scanner.nextLine();
			if (ObjectChecker.isNotEmptyOrNull(line.trim()) && !line.trim().startsWith(LINE_COMMENT))
			{
				int equalIndex = line.indexOf('=');
				if (equalIndex > 0)
				{
					String key = line.substring(0, equalIndex).trim();
					String value = line.substring(equalIndex + 1).trim();
					if (LoggingConfigurator.isInDebugMode())
					{
						if (ObjectChecker.isEmptyOrNull(value))
							fixingWhat.empty(i);
						if (hashMap.containsKey(key))
						{
							boolean areEqual = ObjectChecker.areEqual(hashMap.get(key), value);
							if (areEqual)
								fixingWhat.equally(i);
							else
								fixingWhat.notEqually(i);
						}
					}
					if (spacesReceiver != null && key.contains(" "))
						spacesReceiver.accept(i, line);
					oldIndices.put(key, i);
					hashMap.put(key, value);
					keys.add(key);
				}
				else
					NaMaLogger.error("Bad format while reading from file {0} at line {1}: ({2})", file, i, line);
			}
		}
		scanner.close();
		IOUtils.closeQuietly(resourceAsStream);
		return hashMap;
	}

	@Override
	public String translateMessage(Language lang, String key)
	{
		if (ObjectChecker.isEmptyOrNull(key))
			return null;
		key = key.trim();
		String translation = lang == Language.English ? englishMap.get(key) : arabicMap.get(key);
		return translation;
	}

	@Override
	public String toString()
	{
		return MessageFormat.format("MessageTranslatorFromFile: arabicFile({0})  englishFile({1})", arabicFile, englishFile);
	}

	@Override
	public List<String> getKeys()
	{
		return keys;
	}

	@Override
	public long getOrder()
	{
		return order;
	}

	@Override
	public void processRecursions(AggregatedMessageTranslator aggregatedTranslator)
	{
		this.aggregatedTranslator = aggregatedTranslator;
		processLang(arabicMap, Language.Arabic);
		processLang(englishMap, Language.English);
	}

	@Override
	public void processReplacements(List<MessageTranslator> translators)
	{
		// nothing
	}

	@Override
	public void replace(String key, String value)
	{
		for (Entry<String, String> entry : arabicMap.entrySet())
		{
			if (entry.getValue() == null || key == null || value == null)
				continue;
			entry.setValue(entry.getValue().replace(key, value));
		}
		for (Entry<String, String> entry : englishMap.entrySet())
		{
			if (entry.getValue() == null || key == null || value == null)
				continue;
			entry.setValue(entry.getValue().replace(key, value));
		}
	}

	public Map<String, String> getArabicMap()
	{
		return arabicMap;
	}

	public Map<String, String> getEnglishMap()
	{
		return englishMap;
	}

	public boolean isGeneratedFile()
	{
		return arabicFile.endsWith("-gen.properties") || englishFile.endsWith("-gen.properties");
	}
}
