package com.namasoft.specialserialization;

import com.namasoft.common.utilities.NaMaLogger;
import com.namasoft.common.utilities.StringUtils;

import java.lang.reflect.*;
import java.util.*;

@SuppressWarnings({ "rawtypes", "unchecked" })
public class ReflectionUtils
{
	private static Map<String, Method> allSetterCashedMethods = new HashMap<String, Method>();
	private static Map<String, ArrayList<Method>> allGetterMethods = new HashMap<String, ArrayList<Method>>();

	public static <T> T callMethod(Object object, String methodName, Object... args)
	{
		try
		{
			return callMethod(object, object.getClass().getMethod(methodName), args);
		}
		catch (Exception e)
		{
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	public static <T> T callMethod(Object object, Method method, Object... args)
	{
		Object returnObject;
		try
		{
			returnObject = method.invoke(object, args);
		}
		catch (Exception e)
		{
			e.printStackTrace();
			throw new RuntimeException(e);
		}
		return (T) returnObject;
	}

	public static Object constructObject(String classname) throws Exception
	{
		Class class1 = Class.forName(classname);
		return constructObject(class1);
	}

	public static <T> T constructObject(Class<T> class1) throws Exception
	{
		Constructor<T> constructor = searchForDefaultConstructor(class1);
		if (constructor != null)
			return (T) constructor.newInstance();
		throw new RuntimeException("No Default Constructor Found:" + class1);
	}

	private static Constructor searchForDefaultConstructor(Class class1)
	{
		Constructor[] constructors = class1.getDeclaredConstructors();
		Constructor result = searchForDefaultConstructor(constructors);
		if (result != null)
			return result;
		constructors = class1.getConstructors();
		return searchForDefaultConstructor(constructors);
	}

	protected static Constructor searchForDefaultConstructor(Constructor[] constructors)
	{
		for (int i = 0; i < constructors.length; i++)
		{
			if (constructors[i].getParameterTypes().length == 0)
			{
				if (!constructors[i].isAccessible())
					constructors[i].setAccessible(true);
				return constructors[i];
			}
		}
		return null;
	}

	public static boolean setProperty(Object parent, String propertyName, Object value)
	{
		String key = getPropertyCacheKey(parent, propertyName);
		Method setterMethod = searchForSetterMethod(parent, propertyName, key);
		if (setterMethod == null)
			return false;
		try
		{
			setterMethod.invoke(parent, value);
		}
		catch (Exception e)
		{
			NaMaLogger.error(e);
			return false;
		}
		return true;
	}

	protected static String getPropertyCacheKey(Object parent, String propertyName)
	{
		return parent.getClass() + "." + propertyName;
	}

	protected static Method searchForSetterMethod(Object parent, String propertyName, String key)
	{
		Method setterMethod;
		if (allSetterCashedMethods.containsKey(key))
		{
			setterMethod = allSetterCashedMethods.get(key);
		}
		else
		{
			setterMethod = searchForSetterMethod(parent, propertyName);
			allSetterCashedMethods.put(key, setterMethod);
		}
		return setterMethod;
	}

	private static Method searchForSetterMethod(Object parent, String propertyName)
	{
		Method[] allMethods = parent.getClass().getMethods();
		for (int i = 0; i < allMethods.length; i++)
		{
			if (allMethods[i].getName().equalsIgnoreCase("set" + propertyName) && allMethods[i].getParameterTypes().length == 1)
				return allMethods[i];
		}
		return null;
	}

	public static Class getSuggestedSetterType(Object parent, String propertyName)
	{
		if (parent == null)
			return null;
		Method setterMethod = searchForSetterMethod(parent, propertyName, getPropertyCacheKey(parent, propertyName));
		if (setterMethod == null)
			return null;
		Class class1 = setterMethod.getParameterTypes()[0];
		if (!class1.isInterface() && !Modifier.isAbstract(class1.getModifiers()))
			return class1;
		if (isMap(class1))
			return HashMap.class;
		if (isCollection(class1))
			return ArrayList.class;
		return null;
	}

	static boolean isMap(Class returnType)
	{
		if (Map.class.isAssignableFrom(returnType))
			return true;
		return false;
	}

	static boolean isCollection(Class returnType)
	{
		if (Collection.class.isAssignableFrom(returnType))
			return true;
		return false;
	}

	public static ArrayList<Method> getCahsedGetterMethods(String key)
	{
		return allGetterMethods.get(key);
	}

	public static void setCahsedGetterMethods(String key, ArrayList<Method> methods)
	{
		allGetterMethods.put(key, methods);
	}

	public static void removeAbstractClasses(Collection classes)
	{
		for (Iterator iterator = classes.iterator(); iterator.hasNext(); )
		{
			Class klass = (Class) iterator.next();
			if (Modifier.isAbstract(klass.getModifiers()) || klass.isAnonymousClass())
			{
				iterator.remove();
			}
		}
	}

	public static List<String> classFieldsToList(Class<?> klass)
	{
		Field[] fields = klass.getFields();
		List<String> allFields = new ArrayList<String>();
		for (Field field : fields)
		{
			try
			{
				Object value = field.get(null);
				if (value != null)
					allFields.add(value.toString());
			}
			catch (IllegalArgumentException | IllegalAccessException e)
			{
				NaMaLogger.error(e);
			}
		}
		return allFields;
	}

	public static <T> T getFieldValue(String fieldId, Object root)
	{
		List<String> parts = new ArrayList<>(Arrays.asList(fieldId.split("\\.")));
		Object current = root;
		while (current != null && !parts.isEmpty())
		{
			current = getDirectField(current, parts.remove(0));
		}
		return (T) current;
	}

	private static Object getDirectField(Object current, String singleFieldId)
	{
		Method method = null;
		if (singleFieldId.startsWith("$"))
			method = tryGetMethod(current, singleFieldId.substring(1));
		if (method == null)
			method = tryGetMethod(current, "get" + StringUtils.firstLetterUpper(singleFieldId));
		if (method == null)
			method = tryGetMethod(current, "is" + StringUtils.firstLetterUpper(singleFieldId));
		if (method == null)
			NaMaLogger.error("NOTE: Could not find any getter for the field " + singleFieldId + " in class " + current.getClass());
		method.setAccessible(true);
		try
		{
			return method.invoke(current);
		}
		catch (Exception e)
		{
			throw new RuntimeException(e);
		}
	}

	public static Method tryGetMethod(Object current, String methodName)
	{
		return tryGetMethod(current.getClass(), methodName);
	}

	public static Method tryGetMethod(Class klass, String methodName)
	{
		Method method = null;
		try
		{
			method = klass.getMethod(methodName);
		}
		catch (NoSuchMethodException e)
		{
		}
		return method;
	}

	public static Class getMethodActualReturnType(Method method)
	{
		Type genericReturnType = method.getGenericReturnType();
		if (genericReturnType != null && genericReturnType instanceof ParameterizedType)
		{
			Type[] actualTypeArguments = ((ParameterizedType) genericReturnType).getActualTypeArguments();
			if (actualTypeArguments != null && actualTypeArguments.length > 0)
			{
				Class class1 = (Class) actualTypeArguments[actualTypeArguments.length - 1];
				return class1;
			}
		}
		return method.getReturnType();
	}
}
