package com.namasoft.modules.commonbasic.contracts.entities;
import com.namasoft.common.fieldids.newids.basic.IdsOfEntityTypeList;

import com.namasoft.common.flatobjects.DateFieldUtils;
import com.namasoft.common.flatobjects.EntityReferenceData;
import com.namasoft.common.utilities.DimensionsUtil;
import com.namasoft.common.utilities.ObjectChecker;
import com.namasoft.contracts.common.dtos.DTOHasSubsidiaryAccounts;
import com.namasoft.modules.commonbasic.contracts.details.DTOLegalEntityTax;
import com.namasoft.modules.commonbasic.contracts.valueobjects.DTOGlobalConfigInfo;
import com.namasoft.modules.commonbasic.contracts.valueobjects.DTOTaxConfiguration;
import com.namasoft.modules.commonbasic.enums.TaxConfigDefault;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlRootElement;
import java.io.Serializable;
import java.util.Date;

@SuppressWarnings("serial")
@XmlRootElement
@XmlAccessorType(XmlAccessType.PROPERTY)
public class DTOTaxPlan extends GeneratedDTOTaxPlan implements Serializable, DTOHasSubsidiaryAccounts
{
	public DTOLegalEntityTax findPlan(EntityReferenceData legalEntity)
	{
		if (ObjectChecker.isEmptyOrNull(getLegalEntityTaxes()))
			return null;
		for (DTOLegalEntityTax tax : getLegalEntityTaxes())
		{
			if (ObjectChecker.areEqual(legalEntity, tax.getLegalEntity()) || DimensionsUtil.isNullDimension(tax.getLegalEntity()))
				return tax;
		}
		return null;
	}

	public DTOLegalEntityTax findPlan(EntityReferenceData legalEntity, Date valueDate, String entityType)
	{
		if (ObjectChecker.isEmptyOrNull(getLegalEntityTaxes()))
			return null;
		for (DTOLegalEntityTax tax : getLegalEntityTaxes())
		{
			if (legalEntityMatched(legalEntity, tax) && dateMatched(valueDate, tax.getEffectiveFrom(), tax.getEffectiveTo()) && entityTypeMatched(
					entityType, tax))
				return tax;
		}
		return null;
	}

	private boolean dateMatched(Date valueDate, Date effectiveFrom, Date effectiveTo)
	{

		return ObjectChecker.isEmptyOrNull(valueDate) || DateFieldUtils.isDateBetweenIgnoringNull(valueDate, effectiveFrom, effectiveTo);
	}

	private boolean legalEntityMatched(EntityReferenceData legalEntity, DTOLegalEntityTax tax)
	{
		return ObjectChecker.areEqual(legalEntity, tax.getLegalEntity()) || DimensionsUtil.isNullDimension(tax.getLegalEntity());
	}

	private boolean entityTypeMatched(String entityType, DTOLegalEntityTax tax)
	{
		if (ObjectChecker.areAllEmptyOrNull(tax.getEntityTypeList(), tax.getEntityType()))
			return true;
		if (ObjectChecker.areEqual(tax.getEntityType(), entityType))
			return true;
		if (tax.getEntityTypeList() == null)
			return false;
		if (ObjectChecker.toStringOrEmpty(tax.getEntityTypeList().get(IdsOfEntityTypeList.lines_entityType))
				.contains("," + entityType + ","))
			return true;
		return false;
	}

	public static DTOTaxConfiguration fetchTaxConfiguration(DTOTaxPlan header, DTOTaxPlan line, EntityReferenceData legalEntity, Date valueDate,
			DTOGlobalConfigInfo configInfo, String entityType)
	{
		DTOTaxConfiguration configuration = _fetchTaxConfiguration(header, line, legalEntity, valueDate, configInfo, entityType);

		return configuration;
	}

	public static DTOTaxConfiguration _fetchTaxConfiguration(DTOTaxPlan header, DTOTaxPlan line, EntityReferenceData legalEntity, Date valueDate,
			DTOGlobalConfigInfo configInfo, String entityType)
	{
		if (ObjectChecker.areAllEmptyOrNull(header, line))
			return configInfo.fetchTaxConfig();
		DTOTaxConfiguration taxConfiguration = calcConfigFromPlanOrNull(header, legalEntity, valueDate, configInfo, entityType);
		if (taxConfiguration != null)
			return taxConfiguration;
		taxConfiguration = calcConfigFromPlanOrNull(line, legalEntity, valueDate, configInfo, entityType);
		if (taxConfiguration != null)
			return taxConfiguration;
		return configInfo.fetchTaxConfig();
	}

	private static DTOTaxConfiguration calcConfigFromPlanOrNull(DTOTaxPlan taxPlan, EntityReferenceData legalEntity, Date valueDate,
			DTOGlobalConfigInfo configInfo, String entityType)
	{
		if (taxPlan != null)
		{
			if (ObjectChecker.areEqual(taxPlan.safeGetDefaultTaxConfig(), TaxConfigDefault.GlobalConfig.toString()))
				return configInfo.fetchTaxConfig();
			else if (ObjectChecker.areEqual(taxPlan.safeGetDefaultTaxConfig(), TaxConfigDefault.TaxPlanHeader.toString()))
				return taxPlan.getTaxConfiguration();
			else if (ObjectChecker.areEqual(taxPlan.safeGetDefaultTaxConfig(), TaxConfigDefault.TaxPlanLine.toString()))
			{
				DTOLegalEntityTax plan = taxPlan.findPlan(legalEntity, valueDate, entityType);
				if (plan != null)
					return plan.getTaxConfiguration();
			}
		}
		return null;
	}

	private String safeGetDefaultTaxConfig()
	{
		if (getDefaultTaxConfig() == null)
			return "";
		return getDefaultTaxConfig();
	}
}
