package com.namasoft.common.criteria;

import com.namasoft.common.constants.Operator;
import com.namasoft.common.utilities.*;
import jakarta.xml.bind.annotation.*;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

@XmlAccessorType(XmlAccessType.PROPERTY)
@XmlRootElement
public class DTOCriteria implements Serializable
{
	private static final long serialVersionUID = 1450537244470362116L;
	private static final String OR_EMPTY = "OrEmpty";
	private List<DTOExperssion> expressions;
	private String ownerType;
	private Boolean skipSpecialProcessing = false;
	private Boolean distinctValues;
	private Boolean fromVue;

	public DTOCriteria()
	{
		this(new ArrayList<DTOExperssion>());
	}

	public DTOCriteria(List<DTOExperssion> arrayList)
	{
		expressions = new ArrayList<DTOExperssion>(arrayList);
	}

	public DTOCriteria(String ownerType, List<DTOExperssion> arrayList)
	{
		this(arrayList);
		setOwnerType(ownerType);
	}

	@XmlElementWrapper(name = "experssions")
	@XmlElement(name = "experssion")
	public List<DTOExperssion> getExpressions()
	{
		return expressions;
	}

	public void setExpressions(List<DTOExperssion> experssions)
	{
		this.expressions = experssions;
	}

	public String getOwnerType()
	{
		return ownerType;
	}

	public void setOwnerType(String ownerType)
	{
		this.ownerType = ownerType;
	}

	public Boolean getSkipSpecialProcessing()
	{
		return skipSpecialProcessing;
	}

	public void setSkipSpecialProcessing(Boolean skipSpecialProcessing)
	{
		this.skipSpecialProcessing = skipSpecialProcessing;
	}

	public Boolean getDistinctValues()
	{
		return distinctValues;
	}

	public void setDistinctValues(Boolean distinctValues)
	{
		this.distinctValues = distinctValues;
	}

	public DTOCriteria distinct()
	{
		setDistinctValues(true);
		return this;
	}

	public Boolean getFromVue()
	{
		return fromVue;
	}

	public void setFromVue(Boolean fromVue)
	{
		this.fromVue = fromVue;
	}

	public DTOExperssion getFieldExpression(String fieldId)
	{
		if (ObjectChecker.isNotEmptyOrNull(getExpressions()))
		{
			for (DTOExperssion experssion : getExpressions())
				if (ObjectChecker.areEqual(fieldId, experssion.getField()))
					return experssion;
		}
		return null;
	}

	@Override
	public String toString()
	{
		StringBuilder toString = new StringBuilder().append("owner: ").append(ownerType).append(" expresseions:");
		for (DTOExperssion experssion : getExpressions())
		{
			if (ObjectChecker.isNotEmptyOrNull(experssion))
				toString.append(experssion);
		}
		return toString.toString();
	}

	public DTOCriteria and(DTOCriteria criteria)
	{
		return join(criteria, ExpressionRelationship.AND);
	}

	public DTOCriteria join(DTOCriteria criteria)
	{
		return join(criteria, ExpressionRelationship.AND);
	}

	public DTOCriteria or(DTOCriteria criteria)
	{
		return join(criteria, ExpressionRelationship.OR);
	}

	private DTOCriteria join(DTOCriteria criteria, ExpressionRelationship relationship)
	{
		if (getExpressions() == null)
			setExpressions(new ArrayList<>());
		if (ObjectChecker.isNotEmptyOrNull(criteria) && ObjectChecker.isNotEmptyOrNull(criteria.getExpressions()))
		{
			int size = getExpressions().size();
			if (size > 0)
			{
				getExpressions().add(0, new DTOExperssion("", Operator.OpenBracket, ""));
				DTOExperssion closeBracket = new DTOExperssion("", Operator.CloseBracket, "");
				closeBracket.setRelation(relationship);
				getExpressions().add(closeBracket);
				getExpressions().add(new DTOExperssion("", Operator.OpenBracket, ""));
			}
			getExpressions().addAll(criteria.getExpressions());
			if (size > 0)
			{
				getExpressions().add(new DTOExperssion("", Operator.CloseBracket, ""));
			}
			setDistinctValues(ObjectChecker.isAnyTrue(getDistinctValues(), criteria.getDistinctValues()));
		}
		return this;
	}


	public static DTOCriteria criteriaFromStr(String strExpression)
	{
		if (ObjectChecker.isEmptyOrNull(strExpression))
			return null;
		List<String> expressions = StringUtils.csvLineToList(";|\n|\r", strExpression);
		DTOCriteria criteria = new DTOCriteria();
		for (String expression : expressions)
		{
			String[] parts = expression.split(",");
			if (parts.length > 0 && parts[0] != null && parts[0].trim().equalsIgnoreCase("distinct-values"))
			{
				criteria.setDistinctValues(true);
				continue;
			}
			if (parts.length < 2)
				continue;
			String fieldId = parts[0];
			String operatorStr = parts[1];
			String rightHandSide = parts.length > 2 ? parts[2] : "";
			if (ObjectChecker.toStringOrEmpty(operatorStr).endsWith(OR_EMPTY))
			{
				if (ObjectChecker.isEmptyOrNull(rightHandSide))
					continue;
				operatorStr = operatorStr.substring(0, operatorStr.length() - OR_EMPTY.length());
			}
			Operator operator = ObjectChecker.isEmptyOrNull(operatorStr) ? Operator.Equal : Operator.valueOf(operatorStr);
			ExpressionRelationship relation = parts.length > 3 ? ExpressionRelationship.valueOf(parts[3]) : ExpressionRelationship.AND;
			criteria.getExpressions().add(new DTOExperssion(fieldId, operator, rightHandSide, relation));
		}
		return criteria;
	}

	public String toTextual()
	{
		StringBuilder toString = new StringBuilder();
		for (DTOExperssion experssion : getExpressions())
		{
			if (ObjectChecker.isNotEmptyOrNull(experssion))
			{
				toString.append(experssion.getField()).append(",").append(experssion.getOperator()).append(",").append(experssion.getRightHandSide())
						.append(",").append(experssion.getRelation()).append(";\n");
			}
		}
		return toString.toString();
	}

	public void prefixWith(String prefix)
	{
		if (ObjectChecker.isEmptyOrNull(getExpressions()))
			return;
		for (DTOExperssion x : getExpressions())
		{
			if (ObjectChecker.isNotEmptyOrNull(x.getField()))
				x.setField(prefix + x.getField());
		}
	}

	public DTOCriteria addDefaultCriteriaTo(DTOCriteria criteria)
	{
		if (ObjectChecker.isEmptyOrNull(getExpressions()))
			return criteria;
		if (criteria == null)
			criteria = new DTOCriteria();
		if (criteria.getExpressions() == null)
			criteria.setExpressions(new ArrayList<DTOExperssion>());
		HashSet<String> fieldOperatorSet = new HashSet<>();
		for (DTOExperssion e : criteria.getExpressions())
		{
			fieldOperatorSet.add(e.getField() + e.getOperator());
		}
		for (DTOExperssion e : getExpressions())
		{
			if (!fieldOperatorSet.contains(e.getField() + e.getOperator()))
				criteria.getExpressions().add(e);
		}
		return criteria;
	}

	public static DTOCriteria join(DTOCriteria c1, DTOCriteria c2)
	{
		if (c2 == null)
			return c1;
		if (c1 == null)
			return c2;
		return c1.join(c2);
	}

	public List<DTOExperssion> flattenExpressions()
	{
		if (ObjectChecker.isFalseOrNull(getFromVue()))
			return getExpressions();
		return getExpressions().stream().map(e -> e.flatten(null)).flatMap(Collection::stream).collect(Collectors.toList());
	}
}
