package com.namasoft.common.flatobjects.tempo;

import com.namasoft.common.flatobjects.*;
import com.namasoft.common.utilities.ObjectChecker;

import java.util.*;
import java.util.function.Consumer;

public abstract class RenderNode
{
	protected ComplexRenderer renderer;

	public ComplexRenderer renderer()
	{
		return renderer;
	}

	private final List<RenderNode> nodes = new ArrayList<>();
	private String textPart = "";
	public RenderNode parentNode;
	private Boolean insideCreator = null;

	public RenderNode getParentNode()
	{
		return parentNode;
	}

	public List<RenderNode> getNodes()
	{
		return nodes;
	}

	public List<RenderNode> getNodesForRendering()
	{
		return getNodes();
	}

	public String getTextPart()
	{
		return textPart;
	}

	public void setTextPart(String textPart)
	{
		this.textPart = textPart;
	}

	public void consume(char c)
	{
		textPart += c;
	}

	public void addSubNode(RenderNode subNode)
	{
		subNode.renderer = this.renderer;
		if (isLeafNode())
		{
			parentNode.addSubNode(subNode);
		}
		else
		{
			if (shouldSubNodeEndThis(subNode))
			{
				end("end");
				renderer.currentNode.addSubNode(subNode);
			}
			else
			{
				nodes.add(subNode);
				subNode.parentNode = this;
			}
		}
		renderer.currentNode = subNode;
	}

	public boolean shouldSubNodeEndThis(RenderNode subNode)
	{
		return false;
	}

	public abstract boolean isLeafNode();

	public void renderSubNodes(StringBuilder builder, RendererPropertyResolver helper, int currentLineNumber)
	{
		for (RenderNode node : getNodesForRendering())
		{
			node.render(builder, helper, currentLineNumber);
		}
	}

	public String renderSubNodes(RendererPropertyResolver helper, int currentLineNumber)
	{
		StringBuilder builder = new StringBuilder();
		for (RenderNode node : getNodes())
		{
			node.render(builder, helper, currentLineNumber);
		}
		return builder.toString();
	}

	public boolean canHaveText()
	{
		return false;
	}

	public void end(String nodeContent)
	{
		if (isLeafNode())
		{
			parentNode.end(nodeContent);
		}
		else
		{
			if (ObjectChecker.areEqual(nodeContent, "end") || isMyEnder(nodeContent))
				renderer.currentNode = parentNode;
			else
				parentNode.end(nodeContent);
		}
	}

	public boolean isMyEnder(String nodeContent)
	{
		return false;
	}

	public void toString(StringBuilder builder, int indentationLevel)
	{
		builder.append("\n").append(indent(indentationLevel)).append(describe());
		for (RenderNode node : nodes)
		{
			node.toString(builder, indentationLevel + 1);
		}
		if (!nodes.isEmpty())
			builder.append("\n").append(indent(indentationLevel)).append("end:").append(describe());
	}

	private String indent(int indentationLevel)
	{
		StringBuilder str = new StringBuilder();
		for (int i = 0; i < indentationLevel; i++)
			str.append("\t");
		return str.toString();
	}

	public abstract String describe();

	public abstract void render(StringBuilder builder, RendererPropertyResolver helper, int currentLineNumber);

	public void visit(Consumer<RenderNode> visitor)
	{
		visitor.accept(this);
		for (RenderNode node : getNodes())
		{
			node.visit(visitor);
		}
	}

	protected boolean insideCreator()
	{
		if (insideCreator != null)
			return insideCreator;
		insideCreator = false;
		RenderNode parent = this.parentNode;
		while (parent != null)
		{
			if (parent instanceof CreatorNode)
			{
				insideCreator = true;
				break;
			}
			else if (parent instanceof CreatorMultiValueNode)
			{
				break;
			}
			parent = parent.parentNode;
		}
		return insideCreator;
	}

	public boolean instanceOfOrChildOfType(Class<? extends RenderNode> nodeClass)
	{
		RenderNode current = this;
		while (current != null)
		{
			if (nodeClass.isInstance(current))
				return true;
			current = current.getParentNode();
		}
		return false;
	}

	public List<String> collectVariables()
	{
		return Collections.emptyList();
	}
}
