package com.namasoft.common.criteria;

import com.namasoft.common.constants.Operator;
import com.namasoft.common.flatobjects.*;
import com.namasoft.common.utilities.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;


public class DTOCriteriaBuilder
{
	public static DTOCriteriaBuilder create()
	{
		return new DTOCriteriaBuilder();
	}

	private DTOCriteria criteria;

	private DTOCriteriaBuilder()
	{
		criteria = new DTOCriteria();
	}

	public DTOExpressionBuilder openBracket()
	{
		DTOExpressionBuilder dtoExpressionBuilder = new DTOExpressionBuilder();
		return dtoExpressionBuilder.openBraket();
	}

	public DTOExpressionBuilder field(String id)
	{
		DTOExpressionBuilder dtoExpressionBuilder = new DTOExpressionBuilder();
		return dtoExpressionBuilder.field(id);
	}

	public class DTOExpressionBuilder
	{
		private DTOExperssion expression;

		public DTOExpressionBuilder()
		{
			this(false);
		}

		public DTOExpressionBuilder(boolean dummy)
		{
			if (ObjectChecker.isNotEmptyOrNull(criteria.getExpressions()))
			{
				DTOExperssion last = CollectionsUtility.getLast(criteria.getExpressions());
				if ((last.getOperator() != Operator.OpenBracket && last.getOperator() != Operator.CloseBracket)
						&& ObjectChecker.isEmptyOrNull(last.getField()))
				{
					NaMaLogger.error("You have forgotten to set field id in an expression");
					criteria.getExpressions().remove(last);
				}
			}
			expression = new DTOExperssion();
			relationship(ExpressionRelationship.AND);
			if (!dummy)
				criteria.getExpressions().add(expression);
		}

		public DTOExpressionBuilder field(String id)
		{
			if (ObjectChecker.isNotEmptyOrNull(expression.getField()))
				throw new IllegalArgumentException("field can be only called once on an expression");
			expression.setField(id);
			return this;
		}

		public DTOExpressionBuilder avg(String id)
		{
			expression.setRightHandSide(id);
			expression.setValueType(ExpressionValType.Avg.name());
			return this;
		}

		public DTOExpressionBuilder min(String id)
		{
			expression.setRightHandSide(id);
			expression.setValueType(ExpressionValType.Min.name());
			return this;
		}

		public DTOExpressionBuilder max(String id)
		{
			expression.setRightHandSide(id);
			expression.setValueType(ExpressionValType.Max.name());
			return this;
		}

		public DTOExpressionBuilder theField(String id)
		{
			expression.setRightHandSide(id);
			expression.setValueType(ExpressionValType.AnotherField.name());
			return this;
		}

		public DTOExpressionBuilder contains(String value)
		{
			expression.setOperator(Operator.Contains);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder notContains(String value)
		{
			expression.setOperator(Operator.NotContain);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder startsWith(String value)
		{
			expression.setOperator(Operator.StartsWith);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder endsWith(String value)
		{
			expression.setOperator(Operator.EndsWith);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder notStartsWith(String value)
		{
			expression.setOperator(Operator.NotStartsWith);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder openBraket()
		{
			expression.setOperator(Operator.OpenBracket);
			return new DTOExpressionBuilder();
		}

		public DTOExpressionBuilder closeBraket()
		{
			DTOExpressionBuilder expressionBuilder = expression.getOperator() == null ? this : new DTOExpressionBuilder();
			expressionBuilder.expression.setOperator(Operator.CloseBracket);
			return expressionBuilder;
		}

		@Override
		@Deprecated
		/**
		 * You do not wnat to use this method, you want equal (without s)
		 */
		public boolean equals(Object obj)
		{
			return super.equals(obj);
		}

		public DTOExpressionBuilder equal(boolean b)
		{
			return equal(b ? "true" : "false");
		}

		public <T extends Enum> DTOExpressionBuilder equal(T value)
		{
			if (value == null)
				return equal((String) null);
			else
				return equal(value.name());
		}

		public DTOExpressionBuilder isNull()
		{
			return equal((String) null);
		}

		public DTOExpressionBuilder isNotNull()
		{
			return notEqual((String) null);
		}

		public DTOExpressionBuilder equal(String value)
		{
			expression.setOperator(Operator.Equal);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder compareTo(String fieldId, Operator op)
		{
			expression.setOperator(op);
			expression.setRightHandSide(fieldId);
			expression.setValueType(ExpressionValType.AnotherField.toString());
			return this;
		}

		public DTOExpressionBuilder lessThan(String value)
		{
			expression.setOperator(Operator.LessThan);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder lessThanOrEqual(String value)
		{
			expression.setOperator(Operator.LessThanOrEqual);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder greaterThan(String value)
		{
			expression.setOperator(Operator.GreaterThan);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder greaterThanOrEqual(BigDecimal value)
		{
			return greaterThanOrEqual(FieldTypesUtils.toString(DataTypes.BIGDECIMAL, value));
		}

		public DTOExpressionBuilder lessThanOrEqual(BigDecimal value)
		{
			return lessThanOrEqual(FieldTypesUtils.toString(DataTypes.BIGDECIMAL, value));
		}

		public DTOExpressionBuilder greaterThanOrEqual(String value)
		{
			expression.setOperator(Operator.GreaterThanOrEqual);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder notEqual(String value)
		{
			expression.setOperator(Operator.NotEqual);
			expression.setRightHandSide(value);
			return this;
		}

		public DTOExpressionBuilder or()
		{
			relationship(ExpressionRelationship.OR);
			return new DTOExpressionBuilder();
		}

		public DTOExpressionBuilder and()
		{
			relationship(ExpressionRelationship.AND);
			return new DTOExpressionBuilder();
		}

		public <T extends Enum> DTOExpressionBuilder in(T... values)
		{
			return in(Arrays.stream(values).map(Enum::name).collect(Collectors.toList()));
		}

		public DTOExpressionBuilder in(String... values)
		{
			return in(Arrays.asList(values));
		}

		public DTOExpressionBuilder in(Set<String> values)
		{
			return in(new ArrayList<>(values));
		}

		public DTOExpressionBuilder notIn(Set<String> values)
		{
			return notIn(new ArrayList<>(values));
		}

		public DTOExpressionBuilder in(List<String> values)
		{
			expression.setOperator(Operator.In);
			expression.setInValues(values);
			return this;
		}

		public DTOExpressionBuilder notIn(List<String> values)
		{
			expression.setOperator(Operator.NotIn);
			expression.setInValues(values);
			return this;
		}

		public DTOExpressionBuilder relationship(ExpressionRelationship relationship)
		{
			expression.setRelation(relationship);
			return this;
		}

		public DTOCriteria build()
		{
			return DTOCriteriaBuilder.this.build();
		}

		public DTOExpressionBuilder distinct()
		{
			criteria.setDistinctValues(true);
			return this;
		}

		public DTOExpressionBuilder operator(String operator)
		{
			expression.setOperator(Operator.valueOf(operator));
			return this;
		}

		public DTOExpressionBuilder andCriteria(DTOCriteria criteria)
		{
			relationship(ExpressionRelationship.AND);
			DTOCriteriaBuilder.this.criteria = DTOCriteriaBuilder.this.criteria.join(criteria);
			return this;
		}

		public DTOExpressionBuilder dummy()
		{
			return new DTOExpressionBuilder(true);
		}
	}

	public DTOCriteria build()
	{
		return criteria;
	}

	public static DTOCriteria and(DTOCriteria criteria1, DTOCriteria criteria2)
	{
		if (ObjectChecker.areAllNotEmptyOrNull(criteria1, criteria2)
				&& ObjectChecker.areAllNotEmptyOrNull(criteria1.getExpressions(), criteria2.getExpressions()))
		{
			List<DTOExperssion> experssions = new ArrayList<DTOExperssion>();
			experssions.add(DTOExperssion.openBracket());
			experssions.addAll(criteria1.getExpressions());
			experssions.add(DTOExperssion.closeBracket());
			experssions.add(DTOExperssion.openBracket());
			experssions.addAll(criteria2.getExpressions());
			experssions.add(DTOExperssion.closeBracket());
			DTOCriteria crt = new DTOCriteria(criteria1.getOwnerType(), experssions);
			crt.setDistinctValues(ObjectChecker.isAnyTrue(criteria1.getDistinctValues(), criteria2.getDistinctValues()));
			return crt;
		}
		else if (ObjectChecker.isNotEmptyOrNull(criteria1) && ObjectChecker.isNotEmptyOrNull(criteria1.getExpressions()))
			return criteria1;
		else
			return criteria2;
	}

	public DTOExpressionBuilder dummy()
	{
		return new DTOExpressionBuilder(true);
	}

	public DTOCriteriaBuilder idEqualsValue(String... pairs)
	{
		return idsEqualValues(false, pairs);
	}

	public DTOCriteriaBuilder idsEqualValues(boolean ignoreEmpty, String... pairs)
	{
		if (pairs.length % 2 != 0)
			throw new IllegalArgumentException("Pairs count must even");
		DTOExpressionBuilder experssion = dummy();
		for (int i = 0; i < pairs.length; i += 2)
		{
			if (ignoreEmpty && ObjectChecker.isEmptyOrNull(pairs[i + 1]))
				continue;
			experssion = experssion.and().field(pairs[i]).equal(pairs[i + 1]);
		}
		return this;
	}

	public DTOCriteriaBuilder distinct()
	{
		criteria.setDistinctValues(true);
		return this;
	}
}
