package com.namasoft.common.criteria;

import com.namasoft.common.constants.Operator;
import com.namasoft.common.utilities.*;
import jakarta.xml.bind.annotation.*;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

import static com.namasoft.common.utilities.ObjectChecker.areEqual;
import static com.namasoft.common.utilities.ObjectChecker.compare;

@XmlAccessorType(XmlAccessType.PROPERTY)
@SuppressWarnings("serial")
public class DTOExperssion implements Serializable, NamaObject
{
	private String field;
	private Operator operator;
	private String rightHandSide;
	private List<String> inValues;
	private ExpressionRelationship relation;
	private String valueType;
	private List<DTOExperssion> children;

	public DTOExperssion()
	{

	}

	public DTOExperssion(String fieldId, Operator operator, Object rightHandSide)
	{
		this.field = fieldId;
		this.operator = operator;
		this.rightHandSide = rightHandSide.toString();
		this.relation = ExpressionRelationship.AND;
	}

	public DTOExperssion(String fieldId, Operator operator, Object rightHandSide, ExpressionRelationship expressionRelationship)
	{
		this.field = fieldId;
		this.operator = operator;
		this.rightHandSide = rightHandSide == null ? null : rightHandSide.toString();
		if (rightHandSide != null && "null".equalsIgnoreCase(rightHandSide.toString()))
			this.rightHandSide = null;
		this.relation = expressionRelationship;
	}

	public String getField()
	{
		return field;
	}

	public void setField(String fieldId)
	{
		this.field = fieldId;
	}

	public Operator getOperator()
	{
		return operator;
	}

	public void setOperator(Operator operator)
	{
		this.operator = operator;
	}

	public String getRightHandSide()
	{
		return rightHandSide;
	}

	public void setRightHandSide(String rightHandSide)
	{
		this.rightHandSide = rightHandSide;
	}

	public ExpressionRelationship getRelation()
	{
		return relation;
	}

	public void setRelation(ExpressionRelationship relationship)
	{
		this.relation = relationship;
	}

	@XmlElementWrapper(name = "inValues")
	@XmlElement(name = "inValue")
	public List<String> getInValues()
	{
		return inValues;
	}

	public void setInValues(List<String> inValues)
	{
		this.inValues = inValues;
	}

	public void addInValue(String value)
	{
		if (getInValues() == null)
			setInValues(new ArrayList<String>());
		getInValues().add(value);
	}

	public static DTOExperssion openBracket()
	{
		DTOExperssion experssion = new DTOExperssion();
		experssion.setOperator(Operator.OpenBracket);
		return experssion;
	}

	public static DTOExperssion closeBracket()
	{
		DTOExperssion experssion = new DTOExperssion();
		experssion.setOperator(Operator.CloseBracket);
		experssion.setRelation(ExpressionRelationship.AND);
		return experssion;
	}

	@Override
	public String toString()
	{
		if (operator == Operator.OpenBracket)
			return " (";
		else if (operator == Operator.CloseBracket)
			return ") " + relation;
		else if (operator == Operator.In)
			return field + " " + operator + " '" + StringUtils.toCSVLine(inValues) + "' " + relation;
		return field + " " + operator + " '" + rightHandSide + "' " + relation+" ";
	}

	@Override
	@XmlTransient
	public boolean isEmpty()
	{
		return ObjectChecker.areAllEmptyOrNull(getField(), getInValues(), getOperator(), getRelation(), getRightHandSide());
	}

	public String getValueType()
	{
		return valueType;
	}

	public void setValueType(String valueType)
	{
		this.valueType = valueType;
	}

	@XmlElementWrapper(name = "children")
	@XmlElement(name = "child")
	public List<DTOExperssion> getChildren()
	{
		return children;
	}

	public void setChildren(List<DTOExperssion> children)
	{
		this.children = children;
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	public boolean evalute(Object leftHand, Object rightHand)
	{
		switch (getOperator())
		{
		case Equal:
			return areEqual(leftHand, rightHand);
		case NotEqual:
			return !areEqual(leftHand, rightHand);
		case GreaterThan:
			return compare((Comparable) leftHand, rightHand) > 0;
		case GreaterThanOrEqual:
			return compare((Comparable) leftHand, rightHand) >= 0;
		case LessThan:
			return compare((Comparable) leftHand, rightHand) < 0;
		case LessThanOrEqual:
			return compare((Comparable) leftHand, rightHand) <= 0;
		case StartsWith:
			return toString(leftHand).startsWith(toString(rightHand));
		case NotStartsWith:
			return !toString(leftHand).startsWith(toString(rightHand));
		case EndsWith:
			return toString(leftHand).endsWith(toString(rightHand));
		case NotEndWith:
			return !toString(leftHand).endsWith(toString(rightHand));
		case Contains:
			return toString(leftHand).contains(toString(rightHand));
		case NotContain:
			return !toString(leftHand).contains(toString(rightHand));
		default:
			throw new RuntimeException("Could not perform compare for: " + getOperator());
		}
	}

	private String toString(Object o)
	{
		return ObjectChecker.toStringOrEmpty(o);
	}

	public List<DTOExperssion> flatten(DTOExperssion parent)
	{
		if (ObjectChecker.areNotEqual(getOperator(), Operator.OpenBracket))
			return List.of(this);
		if (ObjectChecker.isEmptyOrNull(getChildren()))
			return List.of();
		List<DTOExperssion> subExpressions = getChildren().stream().map(e -> e.flatten(this)).flatMap(Collection::stream)
				.collect(Collectors.toList());
		if (ObjectChecker.isEmptyOrNull(subExpressions))
			return List.of();
		subExpressions.add(0, this);
		DTOExperssion closeBracket = DTOExperssion.closeBracket();
		closeBracket.setRelation(parent == null ? getRelation() : parent.getRelation());
		subExpressions.add(closeBracket);
		return subExpressions;
	}
}
