package com.namasoft.common.flatobjects;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlTransient;

import com.namasoft.common.fieldids.CommonFieldIds;
import com.namasoft.common.utilities.NamaObject;
import com.namasoft.common.utilities.ObjectChecker;

@XmlAccessorType(XmlAccessType.PROPERTY)
@XmlRootElement
@SuppressWarnings({ "rawtypes", "unchecked" })
public class FlatObject implements Serializable, NamaObject, Cloneable
{
	public FlatRefKey key = FlatRefKey.create();
	private SupportedTypes supportedTypes;
	private static final long serialVersionUID = 7888036813501685527L;
	private String actualClassName = "";
	private HashMap<String, Object> valuesMap = new HashMap<String, Object>();

	public FlatObject(String actualClassName)
	{
		this.actualClassName = actualClassName;
	}

	public FlatObject(Class actualClass)
	{
		this(actualClass.getName());
	}

	public FlatObject()
	{
	}

	public FlatObject(HashMap<String, Object> values)
	{
		setValuesMap(values);
	}

	@XmlTransient
	public ArrayList<String> getAllLeafIds()
	{
		ArrayList<String> ids = new ArrayList<String>();
		for (Entry<String, Object> entry : valuesMap.entrySet())
		{
			String id = entry.getKey();
			Object value = entry.getValue();
			if (isFlatObject(value))
			{
				ids.addAll(((FlatObject) value).getAllLeafIds());
			}
			else
			{
				ids.add(id);
			}
		}
		return ids;
	}

	@XmlTransient
	public ArrayList<String> getAllIds()
	{
		ArrayList<String> ids = new ArrayList<String>();
		for (Entry<String, Object> entry : valuesMap.entrySet())
		{
			ids.add(entry.getKey());
		}
		return ids;
	}

	public boolean isCompositField(String fieldId)
	{
		Object val = getFieldValue(fieldId);
		return isFlatObject(val);
	}

	protected boolean isFlatObject(Object val)
	{
		if (val == null)
			return false;
		if (val instanceof FlatObject)
			return true;
		return false;
	}

	public void setFieldValue(String fieldId, Object value, Integer... indices)
	{
		setFieldValue(fieldId.split("\\."), 0, value, new ArrayList<Integer>(Arrays.asList(indices)));
	}

	private void setFieldValue(String[] keys, int level, Object value, List<Integer> indices)
	{
		if (value instanceof Collection)
		{
			FlatObjectList flatObjectList = new FlatObjectList();
			flatObjectList.setActualClassName(value.getClass().getName());
			flatObjectList.setValue((Collection) value);
			value = flatObjectList;
		}
		if (keys.length == level + 1)
		{
			getValuesMap().put(keys[level], value);
		}
		else
		{
			setInFlatObject(keys, level, value, indices);
		}
	}

	protected void setInFlatObject(String[] keys, int level, Object value, List<Integer> indices)
	{
		Object object = getValuesMap().get(keys[level]);
		boolean putBackToObject = false;
		if (object == null)
		{
			object = new FlatObject();
			putBackToObject = true;
		}
		if (object instanceof FlatObjectList && !isLastPathEntry(keys, level))
		{
			if (containsActualIndices(indices))
			{
				object = ((FlatObjectList) object).getList().get(indices.remove(0));
			}
			else
			{
				ArrayList<Object> list = ((FlatObjectList) object).getList();
				for (Object line : list)
				{
					if (!(line instanceof FlatObject))
					{
						System.err.println("Invalid Object key:" + keys[level] + ", type is: " + line.getClass());
						return;
					}
					((FlatObject) line).setFieldValue(keys, level + 1, value, indices);
				}
				return;
			}
		}
		if (!(object instanceof FlatObject))
		{
			System.err.println("Invalid Object key:" + keys[level] + ", type is: " + object.getClass());
			return;
		}
		((FlatObject) object).setFieldValue(keys, level + 1, value, indices);
		if (putBackToObject)
			getValuesMap().put(keys[level], object);

	}

	public static boolean containsActualIndices(Integer... indices)
	{
		return containsActualIndices(Arrays.asList(indices));
	}

	public static boolean containsActualIndices(List<Integer> indices)
	{
		if (ObjectChecker.isEmptyOrNull(indices))
			return false;
		if (indices.size() == 1 && ObjectChecker.areEqual(indices.get(0), -1))
			return false;
		return true;
	}

	public String getFieldValue_Str(String fieldId, Integer... indices)
	{
		return getFieldValue(fieldId, indices);
	}

	public Integer getFieldValue_Int(String fieldId, Integer... indices)
	{
		return getFieldValue(fieldId, indices);
	}

	public Long getFieldValue_Long(String fieldId, Integer... indices)
	{
		return getFieldValue(fieldId, indices);
	}

	public Boolean getFieldValue_Boolean(String fieldId, Integer... indices)
	{
		return getFieldValue(fieldId, indices);
	}

	public BigDecimal getFieldValue_BigDecimal(String fieldId, Integer... indices)
	{
		return getFieldValue(fieldId, indices);
	}

	public <T> T getFieldValue(String fieldId, Integer... indices)
	{
		if (ObjectChecker.isEmptyOrNull(fieldId))
			return (T) this;
		return (T) getFieldValue(fieldId.split("\\."), 0, new ArrayList<Integer>(Arrays.asList(indices)));
	}

	private Object getFieldValue(String[] fieldPath, int level, List<Integer> indices)
	{
		if (fieldPath.length <= level)
			throw new RuntimeException("invalid field id");
		Object value = valuesMap.get(fieldPath[level]);
		if (value instanceof FlatObjectList && containsActualIndices(indices))
		{
			if (isLastPathEntry(fieldPath, level) && indices.size() == 1)
			{
				return ((FlatObjectList) value).getList().get(indices.remove(0));
			}
			else
			{
				value = ((FlatObjectList) value).getList().get(indices.remove(0));
			}
		}
		if (isLastPathEntry(fieldPath, level))
		{
			return value;
		}
		if (ObjectChecker.areEqual(getActualClassName(), HashMap.class.getName()))
		{
			String newKey = fieldPath[level];
			for (int i = level + 1; i < fieldPath.length; i++)
				newKey += "." + fieldPath[i];
			Object newValue = valuesMap.get(newKey);
			if (ObjectChecker.isNotEmptyOrNull(newValue))
				return newValue;
		}
		if (value == null)
		{
			String concatedId = fieldPath[level];
			int currentLevel = level + 1;
			for (; currentLevel < fieldPath.length; currentLevel++)
			{
				concatedId += "." + fieldPath[currentLevel];
				if (valuesMap.containsKey(concatedId))
				{
					value = valuesMap.get(concatedId);
					if (currentLevel + 1 == fieldPath.length)
						return value;
					if (isFlatObject(value))
						return ((FlatObject) value).getFieldValue(fieldPath, currentLevel + 1, indices);
				}
			}
			// logError("Invalid Object key:" + fieldPath[level]);
			return null;
		}
		if (!isFlatObject(value))
			throw new RuntimeException("invalid field id " + fieldPath[level + 1] + " type: " + value.getClass());
		return ((FlatObject) value).getFieldValue(fieldPath, level + 1, indices);
	}

	private boolean isLastPathEntry(String[] fieldPath, int level)
	{
		return fieldPath.length == (level + 1);
	}

	// private void logError(String error)
	// {
	// NaMaLogger.error(error);
	// }

	public String getActualClassName()
	{
		return actualClassName;
	}

	public void setActualClassName(String realClassName)
	{
		this.actualClassName = realClassName;
	}

	public HashMap<String, Object> getValuesMap()
	{
		return valuesMap;
	}

	void setValuesMap(HashMap<String, Object> values)
	{
		this.valuesMap = values;
	}

	public SupportedTypes getSupportedTypes()
	{
		return supportedTypes;
	}

	public void setSupportedTypes(SupportedTypes supportedTypes)
	{
		this.supportedTypes = supportedTypes;
	}

	@Override
	public boolean isEmpty()
	{
		if (ObjectChecker.isEmptyOrNull(valuesMap))
			return true;
		for (Entry value : valuesMap.entrySet())
		{
			if (ObjectChecker.isNotEmptyOrNull(value.getValue()) && ObjectChecker.areNotEqual(value.getKey(), CommonFieldIds.ID))
				return false;
		}
		return true;
	}

	@Override
	public String toString()
	{
		return "\ntype: " + getActualClassName() + "\n values: " + getValuesMap();
	}

	public FlatObject clone()
	{
		FlatObject clone = new FlatObject();
		clone.setActualClassName(actualClassName);
		for (Entry<String, Object> entry : getValuesMap().entrySet())
		{
			clone.getValuesMap().put(entry.getKey(), CommonFlatObjectUtils.clone(entry.getValue()));
		}
		return clone;
	}

	@Override
	public boolean equals(Object obj)
	{
		if (obj instanceof FlatObject)
		{
			FlatObject other = (FlatObject) obj;
			// if (ObjectChecker.areNotEqual(getActualClassName(),
			// other.getActualClassName()))
			// return false;
			if (!ObjectChecker.areEqualIgnoringEmptyLines(getValuesMap(), other.getValuesMap()))
				return false;
			return true;
		}
		return super.equals(obj);
	}

	@Override
	public int hashCode()
	{
		return getActualClassName().hashCode() ^ getValuesMap().hashCode();
	}

	public FlatObject set(String fieldId, Object value)
	{
		setFieldValue(fieldId, value);
		return this;
	}

	public void copyFieldsTo(FlatObject another, String... fields)
	{
		for (String f : fields)
			another.setFieldValue(f, getFieldValue(f));
	}

	public void copyFieldsToWithPrefix(FlatObject another, String prefix, String... fields)
	{
		for (String f : fields)
			another.setFieldValue(f, getFieldValue(prefix + "." + f));
	}

}
