package com.namasoft.common.flatobjects;

import com.namasoft.common.Pair;
import com.namasoft.common.utilities.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

public class ItemSerialNumberUtil
{
	public static List<String> unZipSerials(String serials)
	{
		if (ObjectChecker.isEmptyOrNull(serials))
			return new ArrayList<>();
		String[] serialRanges = serials.split(getSerialSeperator());
		List<String> retList = new ArrayList<>();
		for (String serialRange : serialRanges)
		{
			if (ObjectChecker.isNotEmptyOrNull(serialRange))
				retList.addAll(unzipSerialRange(serialRange));
		}
		return retList;
	}

	private static List<String> unzipSerialRange(String serialRange)
	{
		List<String> arrayList = new ArrayList<>();
		String[] data = serialRange.split(getSerialRangeSeperator());
		if (data.length > 2)
			throw new RuntimeException("Invalid Serial range " + serialRange);
		if (data.length == 1)
			return Collections.singletonList(serialRange);
		String startSerial = data[0];
		String endSerial = data[1];
		if (notConsistentWith(startSerial, endSerial) || !before(startSerial, endSerial))
			throw new RuntimeException("Invalid Serial range " + serialRange);
		String currentSerial = startSerial;
		while (before(currentSerial, endSerial))
		{
			arrayList.add(currentSerial);
			currentSerial = nextSerial(currentSerial);
		}
		arrayList.add(endSerial);
		return arrayList;
	}

	public static boolean before(String s1, String s2)
	{
		return s1.compareTo(s2) < 0;
	}

	private static String getSerialRangeSeperator()
	{
		return ":";
	}

	private static String getSerialSeperator()
	{
		return ",";
	}

	private static boolean notConsistentWith(String serial, String otherSerial)
	{
		return !consistentWith(serial, otherSerial);
	}

	private static boolean consistentWith(String serial, String otherSerial)
	{
		if (ObjectChecker.areNotEqual(calcPrefix(serial), calcPrefix(otherSerial)))
			return false;
		return calcDecimalPartSize(serial) == calcDecimalPartSize(otherSerial);
	}

	private static int calcDecimalPartSize(String serial)
	{
		if (ObjectChecker.isEmptyOrNull(serial))
			return 0;
		int length = 0;
		for (int i = serial.length() - 1; i >= 0; i--)
		{
			if (!Character.isDigit(serial.charAt(i)))
				return length;
			length++;
		}
		return serial.length();
	}

	private static String calcPrefix(String serial)
	{
		if (ObjectChecker.isEmptyOrNull(serial))
			return "";
		return serial.substring(0, serial.length() - calcDecimalPartSize(serial));
	}

	private static String nextSerial(String serialValue)
	{
		String newSerial = "";
		boolean add = true;
		for (int i = serialValue.length() - 1; i >= 0; i--)
		{
			char charAt = serialValue.charAt(i);
			if (add)
			{

				if (Character.isDigit(charAt))
				{
					int digit = Integer.parseInt("" + charAt);
					if (digit == 9)
						newSerial += "0";
					else
					{
						digit++;
						add = false;
						newSerial = digit + newSerial;
					}
				}
				else
				{
					return null;
				}
			}
			else
			{
				newSerial = charAt + newSerial;
			}
		}
		return newSerial;
	}

	public static String zipSerialRange(List<String> serials)
	{
		return StringUtils.toCSVLineWithSep(",", zipSerialRangeAsList(serials));
	}

	public static List<String> zipSerialRangeAsList(List<String> serials)
	{
		CollectionsUtility.sortAndRemoveDuplicates(serials, Comparator.comparing(ItemSerialNumberUtil::decimalValue));
		List<Pair<String, String>> ranges = new ArrayList<>();
		for (String current : serials)
		{
			Pair<String, String> last = CollectionsUtility.getLast(ranges);
			if (last == null)
			{
				ranges.add(new Pair<>(current, current));
				continue;
			}
			String previousRangeEnd = last.getY();
			if (notConsistentWith(previousRangeEnd, current))
			{
				ranges.add(new Pair<>(current, current));
				continue;
			}
			BigDecimal previousDecimal = decimalValue(previousRangeEnd);
			BigDecimal currentDecimal = decimalValue(current);
			if (currentDecimal.subtract(previousDecimal).intValue() == 1)
				last.setY(current);
			else
				ranges.add(new Pair<>(current, current));
		}
		return ranges.stream().map(r -> ObjectChecker.areEqual(r.getX(), r.getY()) ? r.getX() : r.getX() + getSerialRangeSeperator() + r.getY())
				.collect(Collectors.toList());
	}

	private static BigDecimal decimalValue(String serial)
	{
		String substring = serial.substring(serial.length() - calcDecimalPartSize(serial));
		if (ObjectChecker.isEmptyOrNull(substring))
			return BigDecimal.ZERO;
		return new BigDecimal(substring);
	}
}
