package com.namasoft.common.flatobjects;

import com.namasoft.common.utilities.*;

import java.math.BigDecimal;
import java.time.*;
import java.time.temporal.ChronoUnit;
import java.util.Date;

public class LocalDateUtils
{
	public static LocalDate dateToLocalDate(Date date)
	{
		if (date == null)
			return null;
		if (date instanceof java.sql.Date)
			date = new Date(date.getTime());
		return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
	}

	public static LocalDateTime dateToLocalDateTime(Date date)
	{
		return fetchZonedDateTime(date).toLocalDateTime();
	}

	public static LocalTime dateToLocalTime(Date date)
	{
		return fetchZonedDateTime(date).toLocalTime();
	}

	private static ZonedDateTime fetchZonedDateTime(Date date)
	{
		return Instant.ofEpochMilli(date.getTime()).atZone(ZoneId.systemDefault());
	}

	public static Date localDateToDate(LocalDate localDate)
	{
		return Date.from(localDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
	}

	public static Date localDateTimeToDateTime(LocalDateTime localDateTime)
	{
		return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
	}
	public static long daysBetween(Date start, Date end)
	{
		if (start == null || end == null)
			return 0;
		return ChronoUnit.DAYS.between(dateToLocalDate(start), dateToLocalDate(end));
	}

	public static Date maxDate(Date first, Date second)
	{
		if (first == null)
			return second;
		if (second == null)
			return first;
		return first.getTime() >= second.getTime() ? first : second;
	}

	public static Date dateAfterNDays(Date date, Integer n)
	{
		if (date == null)
			return null;
		if (n == null || n == 0)
			return date;
		return localDateToDate(dateToLocalDate(date).plusDays(n));
	}

	public static boolean areDatesOverLapping(Date s1, Date e1, Date s2, Date e2)
	{
		if (e1 == null)
			e1 = LocalDateUtils.localDateToDate(LocalDate.of(2999, 12, 12));
		if (e2 == null)
			e2 = LocalDateUtils.localDateToDate(LocalDate.of(2999, 12, 12));
		if (s1 == null)
			s1 = LocalDateUtils.localDateToDate(LocalDate.of(1970, 1, 1));
		if (s2 == null)
			s2 = LocalDateUtils.localDateToDate(LocalDate.of(1970, 1, 1));
		return e1.after(s2) && e2.after(s1);
	}

	public static Date addMonthsToDate(int numOfMonths, Date date)
	{
		if (date == null)
			return null;
		return LocalDateUtils.localDateToDate(LocalDateUtils.dateToLocalDate(date).plusMonths(numOfMonths));
	}

	public static BigDecimal minusInHoursSecondsPrecision(Date start, Date end)
	{
		if (ObjectChecker.isAnyEmptyOrNull(start, end))
			return BigDecimal.ZERO;
		LocalDateTime startDate = dateToLocalDateTime(start);
		LocalDateTime endDate = dateToLocalDateTime(end);
		double diffInSeconds = ChronoUnit.SECONDS.between(endDate, startDate);
		return NaMaMath.round(BigDecimal.valueOf(diffInSeconds / 60.0 / 60.0), 6);
	}
}
