package com.namasoft.common.flatobjects;

import com.namasoft.common.ResultDTO;
import com.namasoft.common.constants.CommonConstants;
import com.namasoft.common.layout.edit.NaMaText;
import com.namasoft.common.utilities.*;

import jakarta.xml.bind.annotation.*;
import java.io.Serializable;
import java.util.*;
import java.util.function.Function;

@XmlRootElement
@XmlAccessorType(XmlAccessType.PROPERTY)
public class NaMaError implements Comparable<NaMaError>, Serializable
{
	private NaMaText message;
	private String originalMessage;
	private String rawMessage;
	private String ownerId;
	private Integer row;
	private Boolean messageInHtml;
	private String relatedToOption;
	private String relatedToOptionType;
	private String relatedToOptionCode;
	private String relatedToOptionTypeId;
	private Boolean alwaysShowRelatedTo;
	private Boolean copied = false;
	private String tag;
	private Integer recordIndexInRequest;
	private String recordCode;
	private Map<String, String> extraInfo;

	public static List<NaMaError> createErrorsForLeafResults(ResultDTO result)
	{
		List<NaMaError> errors = new ArrayList<>();
		List<ResultDTO> rootResult = new ArrayList<>();
		rootResult.add(result);
		addErrorsForLeafResults(result, "", "", errors, rootResult);
		return errors;
	}

	public String getRawMessage()
	{
		return rawMessage;
	}

	public void setRawMessage(String rawMessage)
	{
		this.rawMessage = rawMessage;
	}

	private static void addErrorsForLeafResults(ResultDTO result, String prefixMsgAr, String prefixMsgEn, List<NaMaError> errors,
			List<ResultDTO> rootsList)
	{
		if (ObjectChecker.isEmptyOrNull(result))
			return;
		if (isLeafResult(result))
		{
			NaMaError error = new NaMaError();
			if (ObjectChecker.isNotEmptyOrNull(prefixMsgAr))
				prefixMsgAr += "\n";
			if (ObjectChecker.isNotEmptyOrNull(prefixMsgEn))
				prefixMsgEn += "\n";
			error.setMessage(NaMaText.text(prefixMsgAr + result.getArabicMessage(), prefixMsgEn + result.getEnglishMessage()));
			error.setOriginalMessage(result.getOriginalMessage());
			error.setRawMessage(result.getRawMessage());
			error.setOwnerId(getFirsRootWithNotEmptyOwnerId(rootsList, result).getSource());
			error.setRow(getFirsRootWithNotEmptyLineNumber(rootsList, result).getSourceLineNumber());
			error.setRelatedToOption(result.getRelatedToOption());
			error.setRelatedToOptionType(result.getRelatedToOptionType());
			error.setRelatedToOptionTypeId(result.getRelatedToOptionTypeId());
			error.setRelatedToOptionCode(result.getRelatedToOptionCode());
			error.setAlwaysShowRelatedTo(result.getAlwaysShowRelatedTo());
			error.setExtraInfo(result.getExtraInfo());
			if (result.isFailed() || result.isWarning())
				errors.add(error);
		}
		else
		{
			rootsList.add(result);
			if (ObjectChecker.isAnyNotEmptyOrNull(result.getArabicMessage(), result.getEnglishMessage()))
			{
				if (ObjectChecker.isNotEmptyOrNull(prefixMsgAr))
					prefixMsgAr += "\n";
				if (ObjectChecker.isNotEmptyOrNull(prefixMsgEn))
					prefixMsgEn += "\n";
				prefixMsgAr += result.getArabicMessage();
				prefixMsgEn += result.getEnglishMessage();
			}
			for (ResultDTO innerResult : result.getInnerResults())
			{
				addErrorsForLeafResults(innerResult, prefixMsgAr, prefixMsgEn, errors, rootsList);
			}
		}
	}

	private static ResultDTO getFirsRootWithNotEmptyOwnerId(List<ResultDTO> rootsList, ResultDTO result)
	{
		if (hasOwnerId(result))
			return result;
		for (int i = rootsList.size() - 1; i >= 0; i--)
			if (hasOwnerId(rootsList.get(i)))
				return rootsList.get(i);
		return result;
	}

	private static ResultDTO getFirsRootWithNotEmptyLineNumber(List<ResultDTO> rootsList, ResultDTO result)
	{
		if (hasLineNumber(result))
			return result;
		for (int i = rootsList.size() - 1; i >= 0; i--)
			if (hasLineNumber(rootsList.get(i)))
				return rootsList.get(i);
		return result;
	}

	private static boolean hasOwnerId(ResultDTO result)
	{
		return ObjectChecker.isNotEmptyOrNull(result.getSource());
	}

	private static boolean hasLineNumber(ResultDTO result)
	{
		return (result.getSourceLineNumber() != null && result.getSourceLineNumber() > -1);
	}

	private static boolean isLeafResult(ResultDTO result)
	{
		return ObjectChecker.isEmptyOrNull(result.getInnerResults());
	}

	public static NaMaError fromThrowable(Throwable e)
	{
		NaMaError error = new NaMaError();
		if (e.getClass().getName().equals("com.google.gwt.user.client.rpc.StatusCodeException"))
		{
			error.setMessage(NaMaText.resource("couldNotConnectToServer"));
		}
		else if (e.getClass().getName().equals("com.namasoft.erp.gui.base.exceptions.NaMaRPCException")
				|| e.getClass().getName().equals("com.namasoft.contracts.common.uiexceptions.NaMaUIException"))
		{
			try
			{
				error.setMessage(NaMaText.resource(e.getMessage()));
			}
			catch (MissingResourceException exception)
			{
				error.setMessage(NaMaText.text(e.getMessage(), e.getMessage()));
			}
		}
		else if (e.getClass().getName().equals("com.namasoft.erp.gui.base.exceptions.AuthenticationException"))
		{
			error.setMessage(NaMaText.resource(e.getMessage()));
		}
		else if (Arrays.asList("IncompatibleRemoteServiceException", "SerializationException", "SerializedTypeViolationException").contains(
				StringUtils.substringAfterLast(e.getClass().getName(), ".")))
		{
			error.setMessage(NaMaText.resource("refreshPage"));
		}
		else if (e.getMessage() != null && e.getMessage().toLowerCase().startsWith("could not find reference"))
		{
			error.setMessage(NaMaText.text(e.getMessage(),e.getMessage()));
		}
		else
		{
			error.setMessage(NaMaText.resource("technicalError"));
		}
		NaMaLogger.error(e);
		return error;
	}

	public void setMessageInHtml(Boolean messageInHtml)
	{
		this.messageInHtml = messageInHtml;
	}

	public Boolean isMessageInHtml()
	{
		return messageInHtml;
	}

	protected NaMaError()
	{
	}

	public NaMaText getMessage()
	{
		return message;
	}

	public void setMessage(NaMaText message)
	{
		this.message = message;
	}

	public String getOwnerId()
	{
		return ownerId;
	}

	public void setOwnerId(String ownerId)
	{
		this.ownerId = ownerId;
	}

	public Integer getRow()
	{
		return row;
	}

	public void setRow(Integer row)
	{
		this.row = row;
	}

	public static NaMaError error(String message)
	{
		NaMaError error = new NaMaError();
		error.setMessage(NaMaText.text(message, message));
		return error;
	}

	public static NaMaError error(NaMaText message)
	{
		NaMaError error = new NaMaError();
		error.setMessage(message);
		return error;
	}

	public Integer getRecordIndexInRequest()
	{
		return recordIndexInRequest;
	}

	public void setRecordIndexInRequest(Integer recordIndexInRequest)
	{
		this.recordIndexInRequest = recordIndexInRequest;
	}

	public String getRecordCode()
	{
		return recordCode;
	}

	public void setRecordCode(String recordCode)
	{
		this.recordCode = recordCode;
	}

	@Override
	public int compareTo(NaMaError o)
	{
		return compare(getOwnerId(), o.getOwnerId()) + compare(getRow(), o.getRow());
	}

	private <T> int compare(Comparable<T> left, T right)
	{
		if (left == null && right == null)
			return 0;
		if (left == null)
			return -1;
		if (right == null)
			return 1;
		return left.compareTo(right);
	}

	public static List<NaMaError> fromThrowable(Throwable caught, String id, int lineNumber)
	{
		NaMaError error = fromThrowable(caught);
		error.setOwnerId(id);
		error.setRow(lineNumber);
		return Arrays.asList(error);
	}

	public static List<NaMaError> createErrorsForLeafResults(ResultDTO result, String id, int lineNumber)
	{
		ResultDTO wrapperResult = new ResultDTO(null, null, id, null, null, false, false);
		wrapperResult.setSourceLineNumber(lineNumber);
		wrapperResult.setInnerResults(Arrays.asList(result));
		return createErrorsForLeafResults(wrapperResult);
	}

	@Override
	public boolean equals(Object obj)
	{
		if (obj == null || !(obj instanceof NaMaError))
			return false;
		NaMaError otherError = (NaMaError) obj;
		return ObjectChecker.areEqual(getOwnerId(), otherError.getOwnerId()) && ObjectChecker.areEqual(getMessage(), otherError.getMessage())
				&& ObjectChecker.areEqual(getRow(), otherError.getRow());
	}

	@Override
	public int hashCode()
	{
		int hashCode = 0;
		if (getOwnerId() != null)
			hashCode += getOwnerId().hashCode();
		if (getMessage() != null)
			hashCode += getMessage().hashCode();
		if (getRow() != null)
			hashCode += getRow();
		return hashCode;
	}

	public static NaMaError fromString(String message)
	{
		NaMaError error = new NaMaError();
		error.setMessage(NaMaText.text(message, message));
		return error;
	}

	public static NaMaError fromHtml(String message)
	{
		NaMaError error = fromString(message);
		error.setMessageInHtml(true);
		return error;
	}

	public String toParsableStr(NaMaTranslator translator)
	{
		return StringUtils.toCSVLineWithSep(CommonConstants.HOPEFULLY_UNIQUE_FIELD_SEP, getOwnerId(), getRow(), translator.translate(getMessage()));
	}

	public static NaMaError fromParsableString(String error)
	{
		if (error != null && error.contains(CommonConstants.HOPEFULLY_UNIQUE_FIELD_SEP))
		{
			String[] parts = error.split(CommonConstants.HOPEFULLY_UNIQUE_FIELD_SEP_FOR_SPLIT);
			NaMaError naMaError = new NaMaError();
			naMaError.setOwnerId(parts[0]);
			naMaError.setRow(ObjectChecker.valueOfInteger(parts[1]));
			naMaError.setMessage(NaMaText.text(parts[2], parts[2]));
			return naMaError;
		}
		return fromString(error);
	}

	@Override
	public String toString()
	{
		StringBuilder builder = new StringBuilder();
		if (ObjectChecker.isNotEmptyOrNull(getOwnerId()))
			builder.append(getOwnerId()).append(": ");
		if (getRow() != null && getRow() > -1)
			builder.append(" at row: ").append(getRow());
		builder.append(ObjectChecker.getFirstNotEmptyObj(getMessage().getEnglishText(), getMessage().getArabicText(), getMessage().getResourceId()));
		return builder.toString();
	}

	public Boolean isCopied()
	{
		return copied;
	}

	public void setCopied(Boolean copied)
	{
		this.copied = copied;
	}

	public static String toString(List<NaMaError> errors)
	{
		StringBuilder builder = new StringBuilder();
		for (NaMaError naMaError : errors)
		{
			builder.append(naMaError.toString()).append("\n");
		}
		return builder.toString();
	}

	public String getTag()
	{
		return tag;
	}

	public void setTag(String tag)
	{
		this.tag = tag;
	}

	public String getOriginalMessage()
	{
		return originalMessage;
	}

	public void setOriginalMessage(String originalMessage)
	{
		this.originalMessage = originalMessage;
	}

	public String getRelatedToOption()
	{
		return relatedToOption;
	}

	public void setRelatedToOption(String relatedToOption)
	{
		this.relatedToOption = relatedToOption;
	}

	public String getRelatedToOptionType()
	{
		return relatedToOptionType;
	}

	public void setRelatedToOptionType(String relatedToOptionType)
	{
		this.relatedToOptionType = relatedToOptionType;
	}

	public String getRelatedToOptionCode()
	{
		return relatedToOptionCode;
	}

	public void setRelatedToOptionCode(String relatedToOptionCode)
	{
		this.relatedToOptionCode = relatedToOptionCode;
	}

	public String getRelatedToOptionTypeId()
	{
		return relatedToOptionTypeId;
	}

	public void setRelatedToOptionTypeId(String relatedToOptionTypeId)
	{
		this.relatedToOptionTypeId = relatedToOptionTypeId;
	}

	public Boolean getAlwaysShowRelatedTo()
	{
		return alwaysShowRelatedTo;
	}

	public void setAlwaysShowRelatedTo(Boolean alwaysShowRelatedTo)
	{
		this.alwaysShowRelatedTo = alwaysShowRelatedTo;
	}
	public static String describeErrors(List<NaMaError> errors, String separator, Function<NaMaText,String> translator)
	{
		StringBuilder msg = new StringBuilder();
		for (NaMaError error : errors)
		{
			if (msg.length() > 0)
				msg.append(separator);
			if (ObjectChecker.isNotEmptyOrNull(error.getOwnerId()))
				msg.append(error.getOwnerId()).append(": ");
			if (error.getRow() != null && error.getRow() > -1)
				msg.append(" at row: ").append(error.getRow());
			msg.append(translator.apply(error.getMessage()));
		}
		return msg.toString();
	}

	public void setExtraInfo(Map<String, String> extraInfo)
	{
		this.extraInfo = extraInfo;
	}

	public Map<String, String> getExtraInfo()
	{
		return extraInfo;
	}
}
