package com.namasoft.common.flatobjects.tempo;

import com.namasoft.common.exceptions.NaMaServiceExcepption;
import com.namasoft.common.flatobjects.*;
import com.namasoft.common.flatobjects.http.*;
import com.namasoft.common.urlutils.ReportQuestions;
import com.namasoft.common.utilities.*;
import com.namasoft.common.utils.*;
import org.apache.commons.io.IOUtils;

import java.io.*;
import java.math.BigDecimal;
import java.net.*;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.*;
import java.util.function.*;


import java.util.stream.Collectors;

public class ComplexRenderer
{
	public static final String START_ATTACH_DATA_TAG = "<attachData>";
	public static final String END_ATTACH_DATA_TAG = "</attachData>";
	public static final String END_MSG_SUBJECT = "</endMsgSubject>";
	public static final String START_MSG_SUBJECT = "<startMsgSubject>";
	public static final String END_SEND_TO = "</endSendTo>";
	public static final String START_SEND_TO = "<startSendTo>";
	public static final String END_MSG = "</endMsg>";
	public static final String START_NEW_MSG = "<startNewMsg>";
	public static final String OPEN_TABLE = "\n<table border='1' cellspacing='0'>";
	private String text;
	public RenderNode currentNode;
	private RootNode rootNode;
	public final Map<String, Integer> counters = new HashMap<>();
	private boolean directLink = false;
	private boolean shortLinks = false;
	private boolean plainLinks = false;
	private boolean dynamicCriteria = false;
	public String appURL;
	public String originalAppURL;
	public static IComplexRendererGlobalHelper HELPER = null;
	private boolean useCSSFriendlyBrackets = false;
	public boolean shouldClearFactoryFieldPrefixes = false;

	// Registry for nodes with parameters (prefix match, case-insensitive)
	private static final Map<String, Function<String, RenderNode>> NODE_REGISTRY = new LinkedHashMap<>();
	// Registry for simple nodes (exact match, case-insensitive)
	private static final Map<String, Function<String, RenderNode>> SIMPLE_NODE_REGISTRY = new LinkedHashMap<>();
	// Registry for nodes that need renderer reference (receives ComplexRenderer in factory)
	private static final Map<String, Function<ComplexRenderer, RenderNode>> RENDERER_AWARE_NODE_REGISTRY = new LinkedHashMap<>();
	// Registry for nodes with parameters that also need renderer reference (prefix match)
	private static final Map<String, BiFunction<String, ComplexRenderer, RenderNode>> RENDERER_AWARE_PARAM_NODE_REGISTRY = new LinkedHashMap<>();
	// End keywords that trigger currentNode.end()
	private static final Set<String> END_KEYWORDS = Set.of(
			"end", "endtable", "endloop", "endrow", "endlink", "endcell", "endcreator", "endheader",
			"endfooter", "endsendto", "endsubject", "endmsg", "endcomment", "endif", "endelse",
			"endvalue", "endpad", "endreportlink", "endshortenurl", "endmutlivalue"
	);

	static {
		// Nodes with parameters (prefix match)
		NODE_REGISTRY.put("loop(", LoopNode::new);
		NODE_REGISTRY.put("incrementcounter(", IncrementCounterNode::new);
		NODE_REGISTRY.put("decrementcounter(", DecrementCounterNode::new);
		NODE_REGISTRY.put("countervalue(", CounterValueNode::new);
		NODE_REGISTRY.put("padleft(", PadLeftNode::new);
		NODE_REGISTRY.put("padright(", PadRightNode::new);
		NODE_REGISTRY.put("enabledetailsqlfields(", EnableDetailSqlFieldsNode::new);
		NODE_REGISTRY.put("vacationremainder(", VacationRemainder::new);
		NODE_REGISTRY.put("vacationconsumed(", VacationConsumed::new);
		NODE_REGISTRY.put("vacationassigned(", VacationAssigned::new);
		NODE_REGISTRY.put("substring(", SubstringNode::new);
		NODE_REGISTRY.put("left(", LeftNode::new);
		NODE_REGISTRY.put("right(", RightNode::new);
		NODE_REGISTRY.put("header(", GroupHeaderNode::new);
		NODE_REGISTRY.put("footer(", GroupFooterNode::new);
		NODE_REGISTRY.put("creator(", CreatorNode::new);
		NODE_REGISTRY.put("f(", CreatorFieldNode::new);
		NODE_REGISTRY.put("v(", CreatorValueNode::new);
		NODE_REGISTRY.put("callguiaction(", CreatorCallGUIActionNode::new);
		NODE_REGISTRY.put("r(", CreatorRowNode::new);
		NODE_REGISTRY.put("emailattachment(", EmailAttachmentNode::new);
		NODE_REGISTRY.put("recordimage(", RecordImageNode::new);
		NODE_REGISTRY.put("recordimagelink(", RecordImageLinkNode::new);
		NODE_REGISTRY.put("attachmentimage(", AttachmentImageNode::new);
		NODE_REGISTRY.put("attachmentlink(", AttachmentLinkNode::new);
		NODE_REGISTRY.put("titledlink(", TitledLinkNode::new);
		NODE_REGISTRY.put("itemprice(", ItemSalesPriceNode::new);
		NODE_REGISTRY.put("itempurchaseprice(", ItemPurchasePriceNode::new);
		NODE_REGISTRY.put("tafqeet(", TafqeetNode::new);
		NODE_REGISTRY.put("round(", RoundNode::new);
		NODE_REGISTRY.put("formatnumber(", FormatNumberNode::new);
		NODE_REGISTRY.put("formatdate(", FormatDateNode::new);
		NODE_REGISTRY.put("tafqeetar(", TafqeetArNode::new);
		NODE_REGISTRY.put("tafqeeten(", TafqeetEnNode::new);
		NODE_REGISTRY.put("link(", LinkNode::new);
		NODE_REGISTRY.put("flow(", EntityFlowLinkNode::new);
		NODE_REGISTRY.put("onlinepayment(", OnlinePaymentLinkNode::new);
		NODE_REGISTRY.put("time(", TimeNode::new);
		NODE_REGISTRY.put("decimaltotime(", DecimalToTimeNode::new);
		NODE_REGISTRY.put("translate(", TranslateNode::new);
		NODE_REGISTRY.put("getname(", GetNameNode::new);
		NODE_REGISTRY.put("name(", GetNameNode::new);
		NODE_REGISTRY.put("translatear(", TranslateArNode::new);
		NODE_REGISTRY.put("translateen(", TranslateEnNode::new);
		NODE_REGISTRY.put("reportlink(", ReportLinkNode::new);
		NODE_REGISTRY.put("paramname(", ReportLinkParamNameNode::new);
		NODE_REGISTRY.put("paramvalue(", ReportLinkParamValueNode::new);
		NODE_REGISTRY.put("paramrefvalue(", ReportLinkParamRefNode::new);
		NODE_REGISTRY.put("shortenurl(", ShortenURLNode::new);
		NODE_REGISTRY.put("bodypart(", HttpRequestBodyPartNode::new);
		// ListView link nodes
		NODE_REGISTRY.put("listentitytype(", ListViewLinkEntityTypeNode::new);
		NODE_REGISTRY.put("listviewname(", c -> new ListViewLinkPropertyNode(c, "listviewname"));
		NODE_REGISTRY.put("listmenucode(", c -> new ListViewLinkPropertyNode(c, "menucode"));
		NODE_REGISTRY.put("listorderby(", c -> new ListViewLinkPropertyNode(c, "orderby"));
		NODE_REGISTRY.put("listascending(", c -> new ListViewLinkPropertyNode(c, "ascending"));
		NODE_REGISTRY.put("listshowtree(", c -> new ListViewLinkPropertyNode(c, "showtree"));
		NODE_REGISTRY.put("listcurrentpage(", c -> new ListViewLinkPropertyNode(c, "currentpage"));
		NODE_REGISTRY.put("listpagesize(", c -> new ListViewLinkPropertyNode(c, "pagesize"));
		NODE_REGISTRY.put("listextracriteriaid(", c -> new ListViewLinkPropertyNode(c, "extracriteriaid"));

		// Simple nodes (exact match)
		SIMPLE_NODE_REGISTRY.put("tempo", TempoNode::new);
		SIMPLE_NODE_REGISTRY.put("enter", c -> new EnterNode());
		SIMPLE_NODE_REGISTRY.put("openinnewwindow", c -> new OpenInNewWindowNode());
		SIMPLE_NODE_REGISTRY.put("newwindow", c -> new OpenInNewWindowNode());
		SIMPLE_NODE_REGISTRY.put("comment", c -> new CommentNode());

		// Nodes that need renderer reference in constructor (exact match)
		RENDERER_AWARE_NODE_REGISTRY.put("directlinks", DirectLinksNode::new);
		RENDERER_AWARE_NODE_REGISTRY.put("shortlinks", ShortLinksNode::new);
		RENDERER_AWARE_NODE_REGISTRY.put("plainlinks", PlainLinksNode::new);
		RENDERER_AWARE_NODE_REGISTRY.put("dynamiccriteria", DynamicCriteriaNode::new);

		// Nodes with parameters that need renderer reference (prefix match)
		RENDERER_AWARE_PARAM_NODE_REGISTRY.put("appurl(", AppUrlNode::new);
		SIMPLE_NODE_REGISTRY.put("@rownumber", c -> new RowNumberNode());
		SIMPLE_NODE_REGISTRY.put("creatorvalue", CreatorMultiValueNode::new);
		SIMPLE_NODE_REGISTRY.put("table", TableNode::new);
		SIMPLE_NODE_REGISTRY.put("msg", MsgNode::new);
		SIMPLE_NODE_REGISTRY.put("openmsg", c -> new StartMsgNode());
		SIMPLE_NODE_REGISTRY.put("closemsg", c -> new EndMsgNode());
		SIMPLE_NODE_REGISTRY.put("opentable", c -> new OpenTableNode());
		SIMPLE_NODE_REGISTRY.put("closetable", c -> new CloseTableNode());
		SIMPLE_NODE_REGISTRY.put("sendto", SendToNode::new);
		SIMPLE_NODE_REGISTRY.put("subject", SubjectNode::new);
		SIMPLE_NODE_REGISTRY.put("row", TableRowNode::new);
		SIMPLE_NODE_REGISTRY.put("cell", TableCellNode::new);
		SIMPLE_NODE_REGISTRY.put("approvelink", ApproveLinkNode::new);
		SIMPLE_NODE_REGISTRY.put("rejectlink", RejectLinkNode::new);
		SIMPLE_NODE_REGISTRY.put("returnlink", ReturnLinkNode::new);
		SIMPLE_NODE_REGISTRY.put("escalatelink", EscalateToSupervisorLinkNode::new);
		SIMPLE_NODE_REGISTRY.put("returntoprevioussteplink", ReturnToPreviousStepLinkNode::new);
		SIMPLE_NODE_REGISTRY.put("parammultivalue", ReportLinkParamMultiValueNode::new);
		SIMPLE_NODE_REGISTRY.put("httprequest", HttpRequestNode::new);
		SIMPLE_NODE_REGISTRY.put("headername", HttpRequestHeaderNameNode::new);
		SIMPLE_NODE_REGISTRY.put("headervalue", HttpRequestHeaderValueNode::new);
		SIMPLE_NODE_REGISTRY.put("bodypartname", HttpRequestBodyPartNameNode::new);
		SIMPLE_NODE_REGISTRY.put("bodypartvalue", HttpRequestBodyPartValueNode::new);
		SIMPLE_NODE_REGISTRY.put("paramname", HttpRequestParameterNameNode::new);
		SIMPLE_NODE_REGISTRY.put("paramvalue", HttpRequestParameterValueNode::new);
		SIMPLE_NODE_REGISTRY.put("fullbody", HttpRequestBodyNode::new);
		SIMPLE_NODE_REGISTRY.put("requestmethod", HttpRequestMethodNode::new);
		SIMPLE_NODE_REGISTRY.put("contenttype", HttpRequestContentTypeNode::new);
		SIMPLE_NODE_REGISTRY.put("charset", HttpRequestCharSetNode::new);
		SIMPLE_NODE_REGISTRY.put("requestdescription1", HttpRequestDescription1Node::new);
		SIMPLE_NODE_REGISTRY.put("requestdescription2", HttpRequestDescription2Node::new);
		SIMPLE_NODE_REGISTRY.put("requestrelatedtoid1", HttpRequestRelatedToId1Node::new);
		SIMPLE_NODE_REGISTRY.put("requestrelatedtoid2", HttpRequestRelatedToId2Node::new);
		SIMPLE_NODE_REGISTRY.put("requesturl", HttpRequestUrlNode::new);
		// ListView link nodes
		SIMPLE_NODE_REGISTRY.put("listviewlink", ListViewLinkNode::new);
		SIMPLE_NODE_REGISTRY.put("listcriteria", ListViewLinkCriteriaNode::new);
	}

	public ComplexRenderer(String text)
	{
		this.text = text;
	}

	public boolean isDirectLink()
	{
		return directLink;
	}

	public ComplexRenderer directLink(boolean directLink)
	{
		this.directLink = directLink;
		return this;
	}

	public ComplexRenderer dynamicCriteria(boolean dynamicCriteria)
	{
		setDynamicCriteria(dynamicCriteria);
		return this;
	}

	public void shortLinks(boolean shortLinks)
	{
		this.shortLinks = shortLinks;
	}

	public void plainLinks(boolean plainLinks)
	{
		this.plainLinks = plainLinks;
	}

	public boolean isPlainLinks()
	{
		return plainLinks;
	}

	public boolean isDynamicCriteria()
	{
		return dynamicCriteria;
	}

	public void setDynamicCriteria(boolean dynamicCriteria)
	{
		this.dynamicCriteria = dynamicCriteria;
	}

	public static StringBuilder render(String template, StringBuilder result, RendererPropertyResolver resolver)
	{
		if (result == null)
			result = new StringBuilder();
		return result.append(parse(template).render(resolver));
	}

	public boolean isShortLinks()
	{
		return shortLinks;
	}

	public List<String> collectVariables()
	{
		List<String> variables = new ArrayList<>();
		rootNode.visit(n -> variables.addAll(n.collectVariables()));
		return variables.stream().filter(ObjectChecker::isNotEmptyOrNull).filter(s -> !s.contains("\""))
				.filter(s -> ObjectChecker.tryParseDecimal(s) == null).collect(Collectors.toList());
	}

	public List<HttpRequest> httpRequests;

	public List<HttpRequest> getHttpRequests()
	{
		if (httpRequests == null)
			httpRequests = new ArrayList<>();
		return httpRequests;
	}

	public HttpRequest currentRequest;
	public HttpRequestBodyPartHandler currentBodyPartHandler;

	public HttpRequestBodyPartHandler currentBodyPartHandler()
	{
		return currentBodyPartHandler == null ? currentRequest : currentBodyPartHandler;
	}

	public int parseInt(String s)
	{
		return parseDecimal(s).intValue();
	}

	public BigDecimal parseDecimal(String s)
	{
		try
		{
			return new BigDecimal(s);
		}
		catch (Exception e)
		{
			return BigDecimal.ZERO;
		}
	}

	public static boolean isTrue(String s)
	{
		if (s == null || s.isEmpty())
			return false;
		return ObjectChecker.isTrue(s) || ObjectChecker.isTrue(s.replace("\"", "")) || ObjectChecker.isTrue(s.replace("'", ""));
	}

	public static String shortenURL(String server, String signature, String url)
	{
		String shortURL;
		String requestBody = "action=shorturl&url={0}&format=simple&signature={1}";
		if (!server.endsWith("/"))
			server += "/";

		try
		{
			requestBody = MessageFormat.format(requestBody, URLEncoder.encode(url.toString(), StandardCharsets.UTF_8),
					URLEncoder.encode(signature, StandardCharsets.UTF_8));
			URL url1 = new URL(server + "yourls-api.php");
			HttpURLConnection connection = (HttpURLConnection) url1.openConnection();
			connection.setRequestMethod("POST");
			connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
			connection.setRequestProperty("User-Agent", "PostmanRuntime/7.29.0");
			connection.setDoOutput(true);
			try (OutputStream os = connection.getOutputStream())
			{
				os.write(requestBody.getBytes(StandardCharsets.UTF_8));
			}
			int responseCode = connection.getResponseCode();
			if (responseCode == 200 || responseCode == 400)
			{
				//YOURLS returns 400 if the url previously exists, but sends the URL itself in errorStream
				try (InputStream is = responseCode == 200 ? connection.getInputStream() : connection.getErrorStream())
				{
					shortURL = IOUtils.toString(is);
				}
			}
			else
			{
				throw new RuntimeException("Error: " + responseCode);
			}
		}
		catch (IOException e)
		{
			throw new RuntimeException(e);
		}
		return shortURL;
	}

	public String calcGUIURL()
	{
		if (ObjectChecker.isNotEmptyOrNull(appURL))
			return appURL;
		String guiServerURL = NaMaLayersConnector.getInstance().getGuiServerURL();
		if (isDirectLink() || isShortLinks())
			guiServerURL = "";
		return guiServerURL;
	}

	public String evalConstantOrVar(String expression, RendererPropertyResolver helper, int currentLineNumber, boolean insideCreator,
			RenderNode calledFrom)
	{
		if (ObjectChecker.isEmptyOrNull(expression))
			return "";
		try
		{
			new BigDecimal(expression);
			return expression;
		}
		catch (Exception ignored)
		{

		}
		if (expression.startsWith("\""))
			return StringUtils.getInnerString(expression, "\"", "\"");
		if (expression.startsWith("@@") && counters.containsKey(expression.substring("@@".length())))
			return ObjectChecker.toStringOrEmpty(counters.get(expression.substring("@@".length())));
		return helper.resolve(expression, currentLineNumber, insideCreator, calledFrom);
	}

	public Object evalConstantOrVarToObj(String expression, RendererPropertyResolver helper, int currentLineNumber)
	{
		if (ObjectChecker.isEmptyOrNull(expression))
			return null;
		try
		{
			return new BigDecimal(expression);
		}
		catch (Exception ignored)
		{

		}
		if (expression.startsWith("\""))
			return StringUtils.getInnerString(expression, "\"", "\"");
		if (expression.startsWith("@@") && counters.containsKey(expression.substring("@@".length())))
			return counters.get(expression.substring("@@".length()));
		return helper.resolveObject(expression, currentLineNumber);
	}

	public static ComplexRenderer parse(String text)
	{
		return new ComplexRenderer(text).parse();
	}

	public ComplexRenderer parse()
	{
		currentNode = rootNode = new RootNode(this);
		text = ObjectChecker.toStringOrEmpty(text);
		if (text.contains("<notempo/>"))
		{
			TextNode node = new TextNode();
			currentNode.addSubNode(node);
			node.setTextPart(text.replace("<notempo/>", ""));
			return this;
		}
		if (text.contains("<useCSSFriendlyBrackets/>"))
		{
			this.useCSSFriendlyBrackets = true;
			text = text.replace("<useCSSFriendlyBrackets/>", "");
		}
		for (int i = 0; i < text.length(); i++)
		{
			if (text.charAt(i) == '\\')
				i++;
			if (isOpenBracket(i))
			{
				if (useCSSFriendlyBrackets)
					i++;
				int closeBracketIndex = getCloseBracketIndex(i);
				if (closeBracketIndex < 0)
					throw new RuntimeException("Could not find closing bracket in the following text: " + text.substring(i));
				String nodeContent = text.substring(i + 1, closeBracketIndex);
				if (nodeContent.toLowerCase().startsWith("#"))
					nodeContent = nodeContent.substring(1);
				i = closeBracketIndex;
				if (useCSSFriendlyBrackets)
					i++;
				try
				{
					handleNode(nodeContent);
				}
				catch (Exception e)
				{
					Object r1 = HELPER.createFailure("Error while parsing template {0}", text);
					Object r2 = HELPER.createFailure("The part {0} is invalid, please see previous error for the full template",
							"{" + nodeContent + "}");
					throw HELPER.exception(e, r1, r2);
				}
			}
			else
			{
				if (!currentNode.canHaveText())
				{
					TextNode textNode = new TextNode();
					currentNode.addSubNode(textNode);
				}
				currentNode.consume(text.charAt(i));
			}
		}
		return this;
	}

	private int getCloseBracketIndex(int i)
	{
		String closeBracket = "}";
		if (useCSSFriendlyBrackets)
			closeBracket = "}%";
		int closeBracketIndex = text.indexOf(closeBracket, i);
		if (!useCSSFriendlyBrackets && closeBracketIndex > 0)
		{
			if (text.charAt(closeBracketIndex - 1) == '\\')
				return getCloseBracketIndex(closeBracketIndex + 1);
		}
		return closeBracketIndex;
	}

	private boolean isOpenBracket(int i)
	{
		if (useCSSFriendlyBrackets)
		{
			if (i < text.length() - 1 && text.substring(i, i + 2).equals("%{"))
				return true;
			else
				return false;
		}
		if (text.charAt(i) == '{')
		{
			if (i > 0)
			{
				if (text.charAt(i - 1) == '\\')
					return false;
				return true;
			}
			return true;
		}
		return false;
	}

	private void handleNode(String nodeContent)
	{
		String lowerContent = nodeContent.toLowerCase();

		// Handle end nodes - traverse up to find matching node
		if (lowerContent.startsWith("end"))
		{
			if (handleEndNode(nodeContent))
				return;
		}

		assert currentNode != null;

		// Handle else and if nodes (special logic required)
		if (lowerContent.startsWith("else") && handleIfNode(nodeContent, true))
			return;
		if (handleIfNode(nodeContent, false))
			return;

		// Check simple exact matches (case-insensitive)
		Function<String, RenderNode> simpleFactory = SIMPLE_NODE_REGISTRY.get(lowerContent);
		if (simpleFactory != null)
		{
			currentNode.addSubNode(simpleFactory.apply(nodeContent));
			return;
		}

		// Check renderer-aware exact matches (nodes that need renderer in constructor)
		Function<ComplexRenderer, RenderNode> rendererAwareFactory = RENDERER_AWARE_NODE_REGISTRY.get(lowerContent);
		if (rendererAwareFactory != null)
		{
			currentNode.addSubNode(rendererAwareFactory.apply(this));
			return;
		}

		// Check prefix matches for nodes with parameters
		for (Map.Entry<String, Function<String, RenderNode>> entry : NODE_REGISTRY.entrySet())
		{
			if (lowerContent.startsWith(entry.getKey()))
			{
				currentNode.addSubNode(entry.getValue().apply(nodeContent));
				return;
			}
		}

		// Check prefix matches for renderer-aware nodes with parameters
		for (Map.Entry<String, BiFunction<String, ComplexRenderer, RenderNode>> entry : RENDERER_AWARE_PARAM_NODE_REGISTRY.entrySet())
		{
			if (lowerContent.startsWith(entry.getKey()))
			{
				currentNode.addSubNode(entry.getValue().apply(nodeContent, this));
				return;
			}
		}

		// Handle nodes that can be either simple or with parameters (e.g., "approvelink" or "approvelink(...")
		RenderNode hybridNode = tryCreateHybridNode(nodeContent, lowerContent);
		if (hybridNode != null)
		{
			currentNode.addSubNode(hybridNode);
			return;
		}

		// Handle end keywords
		if (END_KEYWORDS.contains(lowerContent))
		{
			currentNode.end(nodeContent);
			return;
		}

		// Default: treat as variable node
		currentNode.addSubNode(new VarNode(nodeContent));
	}

	private boolean handleEndNode(String nodeContent)
	{
		RenderNode node = currentNode;
		while (node != null)
		{
			if (node.isMyEnder(nodeContent))
			{
				node.end(nodeContent);
				return true;
			}
			node = node.getParentNode();
		}
		return false;
	}

	private RenderNode tryCreateHybridNode(String nodeContent, String lowerContent)
	{
		// Nodes that can be used as simple keywords or with parameters
		if (lowerContent.equals("approvelink") || lowerContent.startsWith("approvelink("))
			return new ApproveLinkNode(nodeContent);
		if (lowerContent.equals("rejectlink") || lowerContent.startsWith("rejectlink("))
			return new RejectLinkNode(nodeContent);
		if (lowerContent.equals("returnlink") || lowerContent.startsWith("returnlink("))
			return new ReturnLinkNode(nodeContent);
		if (lowerContent.equals("escalatelink") || lowerContent.startsWith("escalatelink("))
			return new EscalateToSupervisorLinkNode(nodeContent);
		if (lowerContent.equals("returntoprevioussteplink") || lowerContent.startsWith("returntoprevioussteplink("))
			return new ReturnToPreviousStepLinkNode(nodeContent);
		if (lowerContent.equals("listviewlink") || lowerContent.startsWith("listviewlink("))
			return new ListViewLinkNode(nodeContent);
		return null;
	}

	private boolean handleIfNode(String nodeContent, boolean fromElse)
	{
		String contentWithoutElse = nodeContent;
		if (fromElse)
		{
			contentWithoutElse = nodeContent.substring("else".length()).trim();
			currentNode = findFirstParentIfNode(nodeContent);
			if (contentWithoutElse.isEmpty())//simple else
			{
				currentNode.addSubNode(new ElseNode());
				return true;
			}
		}

		if (contentWithoutElse.toLowerCase().startsWith("iflessoreq(") || contentWithoutElse.toLowerCase().startsWith("if<=(")
				|| contentWithoutElse.toLowerCase().startsWith("ifnumberlessoreq(") || contentWithoutElse.toLowerCase().startsWith("ifnumber<=("))
		{
			currentNode.addSubNode(new IfLessOrEqNode(nodeContent));
			return true;
		}
		else if (contentWithoutElse.toLowerCase().startsWith("ifless(") || contentWithoutElse.toLowerCase().startsWith("if<(")
				|| contentWithoutElse.toLowerCase().startsWith("ifnumberless(") || contentWithoutElse.startsWith("ifnumber<("))
		{
			currentNode.addSubNode(new IfLessNode(nodeContent));
			return true;
		}
		else if (contentWithoutElse.toLowerCase().startsWith("ifgreateroreq(") || contentWithoutElse.toLowerCase().startsWith("if>=(")
				|| contentWithoutElse.toLowerCase().startsWith("ifnumbergreateroreq(") || contentWithoutElse.toLowerCase().startsWith("ifnumber>=("))
		{
			currentNode.addSubNode(new IfGreaterOrEqNode(nodeContent));
			return true;
		}
		else if (contentWithoutElse.toLowerCase().startsWith("ifgreater(") || contentWithoutElse.toLowerCase().startsWith("if>(")
				|| contentWithoutElse.toLowerCase().startsWith("ifnumbergreater(") || contentWithoutElse.toLowerCase().startsWith("ifnumber>("))
		{
			currentNode.addSubNode(new IfGreaterNode(nodeContent));
			return true;
		}
		else if (contentWithoutElse.toLowerCase().startsWith("ifequal(") || contentWithoutElse.toLowerCase().startsWith("if=(")
				|| contentWithoutElse.toLowerCase().startsWith("ifnumberequal(") || contentWithoutElse.toLowerCase().startsWith("ifnumber=("))
		{
			currentNode.addSubNode(new IfEqualNode(nodeContent));
			return true;
		}
		else if (contentWithoutElse.toLowerCase().startsWith("ifnotequal(") || contentWithoutElse.toLowerCase().startsWith("if!=(")
				|| contentWithoutElse.toLowerCase().startsWith("ifnumbernotequal(") || contentWithoutElse.toLowerCase().startsWith("ifnumber!=("))
		{
			currentNode.addSubNode(new IfNotEqualNode(nodeContent));
			return true;
		}
		else if (contentWithoutElse.toLowerCase().startsWith("ifnot(") || contentWithoutElse.toLowerCase().startsWith("if!(")
				|| contentWithoutElse.toLowerCase().startsWith("ifnumbernot(") || contentWithoutElse.toLowerCase().startsWith("ifnumber!("))
		{
			currentNode.addSubNode(new IfNotNode(nodeContent));
			return true;
		}
		else if (nodeContent.toLowerCase().startsWith("if(") || nodeContent.toLowerCase().startsWith("ifnumber("))
		{
			currentNode.addSubNode(new IfNode(nodeContent));
		}
		return false;
	}

	private RenderNode findFirstParentIfNode(String nodeContent)
	{
		RenderNode node = currentNode;
		int loops = 0;
		while (nodeDoesNotAcceptElseNodes(node))
		{
			if (node == null)
				break;
			loops++;
			if (loops > 10000)
				throw new NaMaServiceExcepption("Infinite loop while finding parent if statement for the expression " + nodeContent);
			node = node.parentNode;
		}
		if (node == null)
			throw new NaMaServiceExcepption("Could not find parent if statement for the expression " + nodeContent);
		return node;
	}

	private boolean nodeDoesNotAcceptElseNodes(RenderNode node)
	{
		if (!(node instanceof AbstractIfNode ifNode))
			return true;
		if (!ifNode.acceptsElseNodes())
			return true;
		return ifNode.isElse();
	}

	public StringBuilder render(RendererPropertyResolver helper)
	{
		try
		{
			StringBuilder builder = new StringBuilder();
			rootNode.render(builder, helper, 0);
			if (ObjectChecker.isNotEmptyOrNull(originalAppURL))
				HELPER.setGuiServerURL(originalAppURL);
			return builder;
		}
		finally
		{
			if (shouldClearFactoryFieldPrefixes)
				helper.clearFactoryFieldPrefixes();
		}
	}

	@Override
	public String toString()
	{
		StringBuilder builder = new StringBuilder();
		for (RenderNode node : rootNode.getNodes())
		{
			node.toString(builder, 0);
		}
		return builder.toString();
	}

}