package com.namasoft.common.flatobjects.tempo;

import com.namasoft.common.flatobjects.RendererPropertyResolver;
import com.namasoft.common.urlutils.ReportQuestions;
import com.namasoft.common.utilities.*;
import com.namasoft.common.utils.ServerNamaRep;
import com.namasoft.common.utils.translation.TranslationUtil;

import java.util.*;
import java.util.function.Function;

public class ReportLinkNode extends RenderNode
{
	private final String nodeContent;
	private String reportCode;
	private String runType;
	private String plainLink;
	private String title;
	private String targetExpression;

	@Override
	public List<String> collectVariables()
	{
		return Arrays.asList(reportCode, runType, title);
	}

	public ReportLinkNode(String nodeContent)
	{
		this.nodeContent = nodeContent;
		List<String> parts = StringUtils.csvLineToList(StringUtils.getInnerString(nodeContent, "(", ")"));
		for (int i = parts.size() - 1; i >= 0; i--)
		{
			if (parts.get(i).trim().toLowerCase().startsWith("reportcode="))
				reportCode = StringUtils.substringAfter(parts.remove(i), "=");
			else if (parts.get(i).trim().toLowerCase().startsWith("runtype="))
				runType = StringUtils.substringAfter(parts.remove(i), "=");
			else if (parts.get(i).trim().toLowerCase().startsWith("plainlink="))
				plainLink = StringUtils.substringAfter(parts.remove(i), "=");
			else if (parts.get(i).trim().toLowerCase().startsWith("title="))
				title = StringUtils.substringAfter(parts.remove(i), "=");
			else if (parts.get(i).trim().toLowerCase().startsWith("newwindow="))
				targetExpression = StringUtils.substringAfter(parts.remove(i), "=");
		}
	}

	@Override
	public boolean isLeafNode()
	{
		return false;
	}

	@Override
	public String describe()
	{
		return "{" + nodeContent + "}";
	}

	@Override
	public void render(StringBuilder builder, RendererPropertyResolver helper, int currentLineNumber)
	{
		String runType = renderer.evalConstantOrVar(this.runType, helper, currentLineNumber, false, this);
		Object reportCodeOrId = renderer.evalConstantOrVarToObj(this.reportCode, helper, currentLineNumber);
		ReportQuestions questions = ServerNamaRep.repLinkByCode(new HashMap<>(), reportCodeOrId);
		for (RenderNode node : getNodes())
		{
			if (node instanceof IReportLinkSubNode)
			{
				((IReportLinkSubNode) node).render(questions, helper, currentLineNumber);
			}
			else if (node instanceof TextNode)
			{
				//allow text nodes but do not handle them, consider them sort of comment nodes
			}
			else
			{
				throw new UnsupportedOperationException("The node " + node.describe() + " can not be inside " + describe());
			}
		}
		String targetWindow = "";
		if (ObjectChecker.isNotEmptyOrNull(targetExpression))
		{
			if (ObjectChecker.isNotEmptyOrNull(renderer.evalConstantOrVar(targetExpression, helper, currentLineNumber, insideCreator(), this)))
				targetWindow = " target='_blank' ";
		}
		Function<String, String> toLink = questions::toResultLink;
		if (ObjectChecker.areEqual(runType, "launch"))
			toLink = questions::toLaunchLink;
		String titleValue = ObjectChecker.toStringOrEmpty(
				TranslationUtil.translateArgument(TranslationUtil.getCurrentLanguage(), questions.getReportReference()));
		if (ObjectChecker.isNotEmptyOrNull(title))
			titleValue = renderer.evalConstantOrVar(title, helper, currentLineNumber, false, this);
		if (isPlainLink())
			builder.append(toLink.apply(renderer.calcGUIURL()));
		else
			builder.append("<a href='").append(toLink.apply(renderer.calcGUIURL())).append("'").append(targetWindow).append(">").append(titleValue)
					.append("</a>");
	}

	private boolean isPlainLink()
	{
		return ComplexRenderer.isTrue(plainLink);
	}

	@Override
	public boolean isMyEnder(String nodeContent)
	{
		return ObjectChecker.areEqual(nodeContent, "endreportlink");
	}
}
