package com.namasoft.common.utilities;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.*;
import java.util.stream.Collectors;

@SuppressWarnings({ "unchecked", "rawtypes" })
public class CollectionsUtility
{
	public static <T> List<T> listOfNotEmptyObjects(T... ts)
	{
		List<T> list = new ArrayList<>();
		for (T t : ts)
			if (ObjectChecker.isNotEmptyOrNull(t))
				list.add(t);
		return list;
	}
	public static <T> int indexOf(List<T> list, Function<T, Boolean> matcher)
	{
		if (list == null)
			return -1;
		for (int i = 0; i < list.size(); i++)
			if (matcher.apply(list.get(i)))
				return i;
		return -1;
	}

	public static <K, T> Map<K, List<T>> mapList(List<T> list, Function<T, K> keyExtractor)
	{
		return mapList(list, keyExtractor, new HashMap<>());
	}

	public static <K, T> Map<K, List<T>> mapList(List<T> list, Function<T, K> keyExtractor, Map<K, List<T>> map)
	{
		if (ObjectChecker.isEmptyOrNull(list))
			return map;
		for (T l : list)
		{
			getOrAdd(keyExtractor.apply(l), map).add(l);
		}
		return map;
	}

	public static <K, V> List<V> getOrAdd(K key, Map<K, List<V>> map)
	{
		List<V> list = map.get(key);
		if (list == null)
			map.put(key, list = new ArrayList<V>());
		return list;
	}

	public static <T> T getFirstMatching(List<T> list, Filter<T> filter)
	{
		for (T object : list)
		{
			if (filter.include(object))
				return object;
		}
		return null;
	}

	public static <T> List<T> filter(List<T> list, Filter<T> filter)
	{
		if (list == null)
			return new ArrayList<>();
		List<T> retVal = new ArrayList<T>();
		for (T object : list)
		{
			if (filter.include(object))
				retVal.add(object);
		}
		return retVal;
	}

	public static <T> T filterAndReturnFirst(List<T> list, Filter<T> filter)
	{
		return getFirst(filter(list, filter));
	}

	public static <T> T filterAndReturnLast(List<T> list, Filter<T> filter)
	{
		return getLast(filter(list, filter));
	}

	public static <T> void compareTwoLists(List<T> firstList, List<T> secondList, List<T> intersection, List<T> extraInFirst, List<T> extraInSecond)
	{
		compareTwoLists(firstList, secondList, defaultMatcher, intersection, extraInFirst, extraInSecond);
	}

	public static <T> void compareTwoLists(List<T> firstList, List<T> secondList, Matcher<T> matcher, List<T> intersection, List<T> extraInFirst,
			List<T> extraInSecond)
	{
		extraInFirst = newIfNull(extraInFirst);
		extraInSecond = newIfNull(extraInSecond);
		intersection = newIfNull(intersection);
		List<T> twoListIntersection = getIntersection(firstList, secondList, matcher);
		intersection.addAll(twoListIntersection);
		extraInFirst.addAll(subtractTwoLists(firstList, twoListIntersection, matcher));
		extraInSecond.addAll(subtractTwoLists(secondList, twoListIntersection, matcher));
	}

	public static <T> List<T> newIfNull(List<T> list)
	{
		if (list == null)
			list = new ArrayList<T>();
		return list;
	}

	public static <T> List<T> subtractTwoLists(List<T> firstList, List<T> secondList)
	{
		return subtractTwoLists(firstList, secondList, defaultMatcher);
	}

	public static <T> List<T> subtractTwoLists(List<T> firstList, List<T> secondList, Matcher matcher)
	{
		List<T> subtractedList = new ArrayList<T>();
		if (firstList == null || firstList.size() == 0)
			return subtractedList;
		if (secondList == null || secondList.size() == 0)
		{
			subtractedList.addAll(firstList);
			return subtractedList;
		}

		if (matcher == null)
			matcher = defaultMatcher;

		for (T firstObj : firstList)
		{
			boolean found = false;
			for (T secondObj : secondList)
			{
				if (matcher.match(firstObj, secondObj))
				{
					found = true;
					break;
				}
			}
			if (!found)
				subtractedList.add(firstObj);

		}
		return subtractedList;
	}

	public static <T> List<T> getIntersection(List<T> firstList, List<T> secondList)
	{
		return getIntersection(firstList, secondList, null);
	}

	public static <T> List<T> getIntersection(List<T> firstList, List<T> secondList, Matcher matcher)
	{
		return getIntersection(firstList, secondList, matcher, null);
	}

	public static <T> void fillWithLastElement(List<T> list, int newSize)
	{
		while (list.size() < newSize)
			list.add(getLast(list));
	}

	public static <T> Predicate<T> IS_NULL()
	{
		return Objects::isNull;
	}

	public static <T> Predicate<T> IS_EMPTY_OR_NULL()
	{
		return (c) -> ObjectChecker.isEmptyOrNull(c);
	}

	public static <T> BigDecimal totalize(List<T> lines, Function<T, BigDecimal> decimalExtractor)
	{
		if (ObjectChecker.isEmptyOrNull(lines))
			return BigDecimal.ZERO;
		SafeBigDecimal total = SafeBigDecimal.zero();
		for (T l : lines)
			total = total.add(decimalExtractor.apply(l));
		return total.v();
	}

	public static<T> Set<T> join(Set<T> ... sets)
	{
		Set<T> joined = new HashSet<>();
		for (Collection<? extends T> set : sets)
		{
			if (ObjectChecker.isNotEmptyOrNull(set))
				joined.addAll(set);
		}
		return joined;
	}

	public static <T> List<T> copyAndReverse(List<T> list)
	{
		List<T> reversed = new ArrayList<>(list.size());
		for (int i = list.size() - 1; i >= 0; i--)
		{
			reversed.add(list.get(i));
		}
		return reversed;
	}

	public static <T> Set<T> castSet(Set set)
	{
		return set;
	}

	public static interface Callback<T>
	{
		void process(T oldLine, T newLine);
	}

	public static <T> List<T> getIntersection(List<T> firstList, List<T> secondList, Matcher matcher, Callback callback)
	{
		List<T> intersectionResult = new ArrayList<T>();
		if (firstList == null || firstList.size() == 0 || secondList == null || secondList.size() == 0)
			return intersectionResult;
		if (matcher == null)
			matcher = defaultMatcher;
		for (T firstObj : firstList)
		{
			for (T secondObj : secondList)
			{
				if (matcher.match(firstObj, secondObj))
				{
					intersectionResult.add(firstObj);
					if (callback != null)
						callback.process(firstObj, secondObj);
					break;
				}
			}
		}
		return intersectionResult;
	}

	public static <Source, Destination> List<Destination> convert(Collection<? extends Source> source, Converter<Source, Destination> converter)
	{
		if (source == null)
			return new ArrayList<>();

		List<Destination> retVal = new ArrayList<Destination>();
		for (Source object : source)
			retVal.add(converter.convert(object));
		return retVal;
	}

	public static <T> int find(List<T> list, Matcher<T> matcher, T criteria)
	{
		if (ObjectChecker.isEmptyOrNull(list))
			return -1;
		for (int i = 0; i < list.size(); i++)
		{
			if (matcher.match(list.get(i), criteria))
				return i;
		}
		return -1;
	}

	public static List<Integer> getDuplicatesIndices(List list)
	{
		return getDuplicatesIndices(list, defaultMatcher);
	}

	public static List<Integer> getDuplicatesIndices(List list, Matcher matcher)
	{
		List<Integer> duplicateIndices = new ArrayList<Integer>();
		if (matcher == null)
			matcher = defaultMatcher;
		for (int i = 0; i < list.size(); i++)
			for (int j = i + 1; j < list.size(); j++)
			{
				if (matcher.match(list.get(i), list.get(j)))
				{
					duplicateIndices.add(i);
					duplicateIndices.add(j);
				}
			}
		return duplicateIndices;

	}

	public static <T> Matcher<T> defaultMatcher()
	{
		return defaultMatcher;
	}

	private static Matcher defaultMatcher = new Matcher()
	{

		@Override
		public boolean match(Object obj1, Object obj2)
		{
			return ObjectChecker.areEqual(obj1, obj2);
		}
	};

	@SafeVarargs
	public static <K, V> Map<K, V> join(Map<K, V>... maps)
	{
		HashMap<K, V> joined = new HashMap<K, V>();
		for (Map<K, V> map : maps)
		{
			if (ObjectChecker.isNotEmptyOrNull(map))
				joined.putAll(map);
		}
		return joined;
	}

	@SafeVarargs
	public static <T> List<T> join(List<? extends T>... lists)
	{
		List<T> joined = new ArrayList<T>();
		for (Collection<? extends T> list : lists)
		{
			if (ObjectChecker.isNotEmptyOrNull(list))
				joined.addAll(list);
		}
		return joined;
	}

	@SafeVarargs
	public static <T> List<T> join2(List<? extends T>... lists)
	{
		List<T> joined = new ArrayList<T>();
		for (Collection<? extends T> list : lists)
		{
			if (ObjectChecker.isNotEmptyOrNull(list))
				joined.addAll(list);
		}
		return joined;
	}

	public static <T> T getLast(T... ts)
	{
		if (ts == null || ts.length == 0)
			return null;
		return ts[ts.length - 1];
	}

	public static <T> T getFirst(T... ts)
	{
		if (ts == null || ts.length == 0)
			return null;
		return ts[0];
	}

	public static <T> T getLast(List<T> list)
	{
		if (ObjectChecker.isNotEmptyOrNull(list))
			return list.get(list.size() - 1);
		return null;
	}

	public static <T> T getFirst(List<T> list)
	{
		if (ObjectChecker.isNotEmptyOrNull(list))
			return list.get(0);
		return null;
	}

	public static <T> T removeFirst(List<T> list)
	{
		if (ObjectChecker.isNotEmptyOrNull(list))
			return list.remove(0);
		return null;
	}

	public static <T> T getFirstNotEmptyOrNull(List<T> list)
	{
		for (T element : list)
		{
			if (ObjectChecker.isNotEmptyOrNull(element))
				return element;
		}
		return null;
	}

	public static <T> T[] join(T[]... p)
	{
		if (ObjectChecker.isEmptyOrNull(p))
			return null;
		List<T> values = new ArrayList<T>();
		for (int i = 0; i < p.length; i++)
		{
			if (p[i] != null)
				values.addAll(Arrays.asList(p[i]));
		}
		return values.toArray(p[0]);
	}

	public static <T> T[] array(T... array)
	{
		return array;
	}

	public static <T> T[] toArray(Collection<T> collection)
	{
		return collection.toArray(CollectionsUtility.<T> array());
	}

	public static <T> List<T> castList(List list)
	{
		return list;
	}

	public static <T> List<T> asList(T... ts)
	{
		if (ts == null)
			return new ArrayList<T>();
		return new ArrayList<T>(Arrays.asList(ts));
	}

	public static <T extends Comparable<? super T>> List<T> sort(List<T> list)
	{
		Collections.sort(list);
		return list;
	}

	public static boolean listsAreIdentical(List<?> l1, List<?> l2)
	{
		if (ObjectChecker.areAllEmptyOrNull(l1, l2))
			return true;
		if (ObjectChecker.isAnyEmptyOrNull(l1, l2))
			return false;
		if (ObjectChecker.areNotEqual(l1.size(), l2.size()))
			return false;
		for (int i = 0; i < l1.size(); i++)
		{
			if (l1.get(i) != l2.get(i))
				return false;
		}
		return true;
	}

	public static <T1, T2> void makeSecondSameSize(List<T1> first, List<T2> second, ObjectCreator<T2> creator)
	{
		if (first == null)
			first = Collections.emptyList();
		makeSize(first.size(), second, creator);
	}

	public static <T1, T2> void makeSize(int size, List<T2> second, ObjectCreator<T2> creator)
	{
		for (int i = second.size(); i < size; i++)
		{
			second.add(creator.create());
		}
		for (int i = second.size() - 1; i > size - 1; i--)
		{
			second.remove(i);
		}
	}

	public static int smartFind(List details, Object criteria, int start, Matcher matcher)
	{
		if (ObjectChecker.isEmptyOrNull(details))
			return -1;
		if (start < 0)
			start = 0;
		if (start > details.size())
			start = details.size();
		for (int i = start; i < details.size(); i++)
		{
			if (matcher.match(details.get(i), criteria))
				return i;
		}
		for (int i = 0; i < start; i++)
		{
			if (matcher.match(details.get(i), criteria))
				return i;
		}
		return -1;
	}

	@SafeVarargs
	public static <T> List<T> toList(T... params)
	{
		List<T> ret = new ArrayList<T>();
		if (params == null)
			return ret;
		for (int i = 0; i < params.length; i++)
			ret.add(params[i]);
		return ret;
	}

	public static <T extends Comparable<T>> void sortAndRemoveDuplicates(List<T> list)
	{
		sortAndRemoveDuplicates(list, Comparator.naturalOrder());
	}

	public static <T> void sortAndRemoveDuplicates(List<T> list, Comparator<T> comparator)
	{
		Collections.sort(list, comparator);
		for (int i = list.size() - 1; i > 0; i--)
		{
			if (list.get(i).equals(list.get(i - 1)))
				list.remove(i);
		}
	}

	public static <K, V> Map<K, V> map(Object... objects)
	{
		HashMap<K, V> map = new HashMap<K, V>();
		for (int i = 0; i < objects.length; i += 2)
		{
			map.put((K) objects[i], (V) objects[i + 1]);
		}
		return map;
	}

	@SafeVarargs
	public static <T> Set<T> hashSet(T... objects)
	{
		return new HashSet<T>(Arrays.asList(objects));
	}

	public static void fixMasterRowIds(List<? extends IHasMasterRowId> original, List<? extends IHasMasterRowId> copy, IDProvider idProvider)
	{
		HashMap<Object, Object> oldNewIdsMap = new HashMap<Object, Object>();
		for (int i = 0; i < original.size() && i < copy.size(); i++)
		{
			if (idProvider != null)
				copy.get(i).doSetId(idProvider.newId());
			oldNewIdsMap.put(original.get(i).fetchId(), copy.get(i).fetchId());
		}
		for (IHasMasterRowId line : copy)
		{
			if (ObjectChecker.isNotEmptyOrNull(line.fetchMasterRowId()))
				line.doSetMasterRowId(oldNewIdsMap.get(line.fetchMasterRowId()));
		}
	}

	public static <T> T removeFirstOrCreateNew(List<T> list, ObjectCreator<T> creator)
	{
		if (ObjectChecker.isEmptyOrNull(list))
			return creator.create();
		T remove = list.remove(0);
		if (remove == null)
			return removeFirstOrCreateNew(list, creator);
		return remove;
	}

	public static <T> ObjectCreator<T> getFirstOrCreateNewCreator(List<T> list, ObjectCreator<T> creator)
	{
		List<T> copy = new ArrayList<>(list);
		return () -> removeFirstOrCreateNew(copy, creator);
	}

	public static <T> ObjectCreator<T> removeFirstOrCreateNewCreator(List<T> list, ObjectCreator<T> creator)
	{
		return () -> removeFirstOrCreateNew(list, creator);
	}

	public static <T, V> void setInDetails(Collection<T> list, BiConsumer<T, V> setter, V v)
	{
		if (list == null)
			return;
		for (T t : list)
		{
			if (t == null)
				continue;
			setter.accept(t, v);
		}
	}

	public static <T, V> void setInDetails(Collection<T> list, BiConsumer<T, V> setter, V v, Predicate<T> filter)
	{
		if (list == null)
			return;
		for (T t : list)
		{
			if (t == null)
				continue;
			if (filter.test(t))
			{
				setter.accept(t, v);
			}
		}
	}

	public static <T> void addToList(List<T> list, int index, T item)
	{
		if (index < 0 || index >= list.size())
		{
			list.add(item);
			return;
		}
		list.add(index, item);
	}

	public static <R, T> Set<R> mapToSet(Collection<T> lines, Function<T, R> supplier)
	{
		return lines.stream().map(supplier).collect(Collectors.toSet());
	}

	public static <T, R> Function<T, R> cast(Class<R> c)
	{
		return (o) -> (R) o;
	}
}
