package com.namasoft.common.utilities;

import com.namasoft.common.constants.*;
import com.namasoft.common.fieldids.newids.basic.IdsOfCurrency;
import com.namasoft.common.flatobjects.*;

import java.math.*;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Stream;

public class NaMaMath
{
	public static BigDecimal OneHundered = new BigDecimal("100");

	public static BigDecimal getMax(BigDecimal firstNumber, BigDecimal secondNumber)
	{
		if (ObjectChecker.isEmptyOrNull(firstNumber))
			firstNumber = BigDecimal.ZERO;
		if (ObjectChecker.isEmptyOrNull(secondNumber))
			secondNumber = BigDecimal.ZERO;
		return firstNumber.max(secondNumber);
	}

	public static BigDecimal getMin(BigDecimal... numbers)
	{
		if (numbers.length == 0)
			return BigDecimal.ZERO;
		BigDecimal min = numbers[0];
		if (min == null)
			min = BigDecimal.ZERO;
		for (int i = 1; i < numbers.length; i++)
		{
			BigDecimal n = numbers[i];
			if (n == null)
				n = BigDecimal.ZERO;
			min = min.min(n);
		}
		return min;
	}

	public static BigDecimal divideByCurrency(BigDecimal bast, BigDecimal maqam, EntityReferenceData currency)
	{
		return divide(bast, maqam, scaleOfCurrency(currency));
	}

	public static BigDecimal divide(BigDecimal bast, BigDecimal maqam, Integer scale)
	{
		if (scale == null)
			scale = 0;
		if (ObjectChecker.isEmptyOrZero(maqam))
			return BigDecimal.ZERO;
		if (ObjectChecker.isEmptyOrZero(bast))
			return BigDecimal.ZERO;
		return bast.divide(maqam, scale, CommonConstants.DEFAULT_ROUNDING_MODE);
	}

	public static BigDecimal  round(BigDecimal value, Integer scale)
	{
		value = ObjectChecker.toZeroIfNull(value);
		if (scale == null || scale == 0)
		{
			scale = 0;
			BigDecimal noFactionsPart = divide(value, BigDecimal.ONE, scale);
			BigDecimal diff = value.subtract(noFactionsPart);
			if (diff.compareTo(new BigDecimal(0.5)) >= 0)
				return noFactionsPart.add(BigDecimal.ONE);
			return noFactionsPart;
		}
		if (scale < 0)
			return value;
		return value.setScale(scale, CommonConstants.DEFAULT_ROUNDING_MODE);
	}

	public static BigDecimal oneIfEmptyOrZero(BigDecimal value)
	{
		if (ObjectChecker.isEmptyOrNull(value) || ObjectChecker.isZero(value))
			return BigDecimal.ONE;
		return value;
	}

	public static BigDecimal toBigDecimal(Object fieldValue)
	{
		if (ObjectChecker.isEmptyOrNull(fieldValue))
			return BigDecimal.ZERO;
		if (fieldValue instanceof BigDecimal)
			return (BigDecimal) fieldValue;
		if (fieldValue instanceof Integer)
			return new BigDecimal((Integer) fieldValue);
		if (fieldValue instanceof Long)
			return new BigDecimal((Long) fieldValue);
		return new BigDecimal(fieldValue.toString());
	}

	public static BigDecimal calcTimeRate(String from, BigDecimal fromValue, String to)
	{
		return calcTimeRate(from, to).multiply(fromValue);
	}

	public static BigDecimal calcTimeRate(String from, String to)
	{
		BigDecimal fromInSeconds = calcSecondsRate(toTimePeriodEnum(from));
		BigDecimal toInSeconds = calcSecondsRate(toTimePeriodEnum(to));
		return toInSeconds.divide(fromInSeconds, 6, RoundingMode.UP);
	}

	private static BigDecimal calcSecondsRate(TimePeriodType timePeriodEnum)
	{
		switch (timePeriodEnum)
		{
		case Second:
			return new BigDecimal("1");
		case Minute:
			return new BigDecimal("60");
		case Hour:
			return new BigDecimal("3600");
		case Day:
			return new BigDecimal("86400");
		case Week:
			return new BigDecimal("604800");
		case Month:
			return new BigDecimal("2592000");
		case Year:
			return new BigDecimal("31104000");
		default:
			return new BigDecimal("0");
		}

	}

	private static TimePeriodType toTimePeriodEnum(String enumStr)
	{
		try
		{
			return TimePeriodType.valueOf(enumStr);
		}
		catch (Exception e)
		{
			e.printStackTrace();
			return null;
		}
	}

	public static BigDecimal calcPercentageValue(BigDecimal total, BigDecimal percentage)
	{
		return calcPercentageValue(total, percentage, 10);
	}

	public static BigDecimal calcPercentageValue(BigDecimal total, BigDecimal percentage, EntityReferenceData currency)
	{
		return calcPercentageValue(total, percentage, scaleOfCurrency(currency));
	}

	private static Integer scaleOfCurrency(EntityReferenceData currency)
	{
		return currency == null ? 10 : currency.get(IdsOfCurrency.fractionDecimalPlaces);
	}

	public static BigDecimal calcPercentageValue(BigDecimal total, BigDecimal percentage, Integer decimalPlaces)
	{
		total = zeroIfNull(total);
		percentage = zeroIfNull(percentage);
		return percentage.multiply(total).divide(OneHundered, decimalPlaces, CommonConstants.DEFAULT_ROUNDING_MODE);
	}

	public static BigDecimal zeroIfNull(BigDecimal value)
	{
		return ObjectChecker.toZeroIfNull(value);
	}

	//	public static BigDecimal zeroIfNull(Number value)
	//	{
	//		if (value == null)
	//			return BigDecimal.ZERO;
	//		if (value instanceof BigDecimal)
	//			return (BigDecimal) value;
	//		if (value instanceof Integer)
	//			return new BigDecimal((Integer) value);
	//		if (value instanceof Double)
	//			return new BigDecimal((Double) value);
	//		if (value instanceof Long)
	//			return new BigDecimal((Long) value);
	//		throw new RuntimeException("Unhandled value in NamaMath.zeroIfNull" + value);
	//	}
	public static BigDecimal calcPercentage(BigDecimal total, BigDecimal discountValue)
	{
		return calcPercentage(total, discountValue, 7);
	}

	public static BigDecimal calcPercentage(BigDecimal total, BigDecimal discountValue, Integer scale)
	{
		discountValue = zeroIfNull(discountValue);
		total = zeroIfNull(total);
		if (total.compareTo(BigDecimal.ZERO) == 0)
			return BigDecimal.ZERO;
		BigDecimal value = discountValue.multiply(OneHundered).divide(total, scale, CommonConstants.DEFAULT_ROUNDING_MODE);
		return value;
	}

	public static void addToBigDecimalInFlat(FlatObject value, BigDecimal toAdd, String valueId)
	{
		BigDecimal totalvalue = zeroIfNull(value.<BigDecimal>getFieldValue(valueId));
		value.setFieldValue(valueId, totalvalue.add(toAdd));
	}

	public static boolean betweenExcludingUpper(BigDecimal value, BigDecimal min, BigDecimal max)
	{
		if (isLessThan(value, min))
			return false;
		if (isGreaterThanOrEquals(value, max))
			return false;
		return true;
	}

	public static boolean between(BigDecimal value, BigDecimal min, BigDecimal max)
	{
		if (isLessThan(value, min))
			return false;
		if (isGreaterThan(value, max))
			return false;
		return true;
	}

	public static boolean between(Integer value, Integer min, Integer max)
	{
		if (isLessThan(value, min))
			return false;
		if (isGreaterThan(value, max))
			return false;
		return true;
	}

	public static boolean betweenIgnoringZero(Integer value, Integer min, Integer max)
	{
		if (ObjectChecker.isNotEmptyOrZero(min) && isLessThan(value, min))
			return false;
		if (ObjectChecker.isNotEmptyOrZero(max) && isGreaterThan(value, max))
			return false;
		return true;
	}

	public static boolean probableBetween(BigDecimal value, BigDecimal min, BigDecimal max)
	{
		if (value.add(BIGDECIMALMARGIN).subtract(min).compareTo(BigDecimal.ZERO) < 0)
			return false;
		if (max.add(BIGDECIMALMARGIN).subtract(value).compareTo(BigDecimal.ZERO) < 0)
			return false;
		return true;
	}

	public static boolean notBetween(BigDecimal value, BigDecimal min, BigDecimal max)
	{
		return !probableBetween(value, min, max);
	}

	public static final BigDecimal BIGDECIMALMARGIN = new BigDecimal("0.001");
	public static final BigDecimal MinBIGDECIMALMARGIN = new BigDecimal("0.0002");

	public static boolean isPositive(BigDecimal qty)
	{
		if (qty == null)
			return false;
		return qty.compareTo(BigDecimal.ZERO) > 0;
	}

	public static boolean isNegative(BigDecimal qty)
	{
		if (qty == null)
			return false;
		return qty.compareTo(BigDecimal.ZERO) < 0;
	}

	public static boolean probableZero(BigDecimal n)
	{
		return probableZero(n, MinBIGDECIMALMARGIN);
	}

	public static boolean probableZero(BigDecimal n, BigDecimal ignoredValue)
	{
		if (n == null)
			return true;
		return n.abs().compareTo(ignoredValue) < 0;
	}

	public static boolean isLessThan(BigDecimal firstValue, BigDecimal secondValue)
	{
		return zeroIfNull(firstValue).compareTo(zeroIfNull(secondValue)) < 0;
	}

	public static boolean isLessThan(Integer firstValue, Integer secondValue)
	{
		return ObjectChecker.toZeroIfNull(firstValue).compareTo(ObjectChecker.toZeroIfNull(secondValue)) < 0;
	}

	public static boolean isGreaterThan(BigDecimal firstValue, BigDecimal secondValue)
	{
		return zeroIfNull(firstValue).compareTo(zeroIfNull(secondValue)) > 0;
	}

	public static boolean isGreaterThan(Integer firstValue, Integer secondValue)
	{
		return ObjectChecker.toZeroIfNull(firstValue).compareTo(ObjectChecker.toZeroIfNull(secondValue)) > 0;
	}

	public static boolean isLessThanOrEquals(BigDecimal firstValue, BigDecimal secondValue)
	{
		return zeroIfNull(firstValue).compareTo(zeroIfNull(secondValue)) <= 0;
	}

	public static boolean isGreaterThanOrEquals(BigDecimal firstValue, BigDecimal secondValue)
	{
		return zeroIfNull(firstValue).compareTo(zeroIfNull(secondValue)) >= 0;
	}

	public static boolean isLessThanOrEqual(BigDecimal firstValue, BigDecimal secondValue)
	{
		BigDecimal first = zeroIfNull(firstValue);
		BigDecimal second = zeroIfNull(secondValue);
		return first.compareTo(second) < 0 || first.compareTo(second) == 0;
	}

	public static String toString(BigDecimal value, int scale)
	{
		value = NaMaMath.round(value, scale);
		return value.stripTrailingZeros().toPlainString();
	}

	public static int max(int... values)
	{
		if (values.length == 0)
			return 0;
		int max = values[0];
		for (int n : values)
		{
			max = Math.max(max, n);
		}
		return max;
	}

	public static int min(int... values)
	{
		if (values.length == 0)
			return 0;
		int min = values[0];
		for (int n : values)
		{
			min = Math.min(min, n);
		}
		return min;
	}

	public static BigDecimal multiplyNonZero(BigDecimal... numbers)
	{
		BigDecimal result = BigDecimal.ZERO;
		for (BigDecimal n : numbers)
		{
			if (ObjectChecker.isEmptyOrZero(n))
				continue;
			if (ObjectChecker.isEmptyOrZero(result))
				result = BigDecimal.ONE;
			result = multiply(result, n);
		}
		return result;
	}

	public static int multiplyNonZero(int... numbers)
	{
		int result = 0;
		for (int n : numbers)
		{
			if (n == 0)
				continue;
			if (result == 0)
				result = 1;
			result *= n;
		}
		return result;
	}

	public static BigDecimal multiply(BigDecimal... numbers)
	{
		BigDecimal total = BigDecimal.ONE;
		for (BigDecimal n : numbers)
		{
			if (n == null)
				return BigDecimal.ZERO;
			total = total.multiply(n);
		}
		return total;
	}

	public static BigDecimal add(BigDecimal... numbers)
	{
		BigDecimal total = BigDecimal.ZERO;
		for (BigDecimal n : numbers)
		{
			if (n != null)
				total = total.add(n);
		}
		return total;
	}

	public static BigDecimal remainder(BigDecimal bast, BigDecimal maqam)
	{
		if (ObjectChecker.isEmptyOrZero(bast))
			return BigDecimal.ZERO;
		if (ObjectChecker.isEmptyOrZero(maqam))
			return BigDecimal.ZERO;
		return bast.remainder(maqam, MathContext.DECIMAL64);
	}

	public static BigDecimal sum(BigDecimal... values)
	{
		SafeBigDecimal sum = SafeBigDecimal.zero();
		for (BigDecimal d : values)
			sum = sum.add(d);
		return sum.v();
	}

	public static Integer sum(int... values)
	{
		Integer sum = 0;
		for (Integer d : values)
			sum += d;
		return sum;
	}

	public static Integer sum(Integer... values)
	{
		Integer sum = 0;
		for (Integer d : values)
			sum += d == null ? 0 : d;
		return sum;
	}

	public static BigDecimal subtract(BigDecimal arg1, BigDecimal arg2)
	{
		arg1 = zeroIfNull(arg1);
		arg2 = zeroIfNull(arg2);
		return arg1.subtract(arg2);
	}

	public static BigDecimal roundValue(BigDecimal value, BigDecimal roundingValue, String roundingType)
	{
		return roundValue(value, roundingValue, roundingType, 10);
	}

	public static BigDecimal roundValue(BigDecimal value, BigDecimal roundingValue, String roundingType, Integer fractionalDecimalPlaces)
	{
		if (fractionalDecimalPlaces == null)
			fractionalDecimalPlaces = 0;
		if (ObjectChecker.isEmptyOrNull(roundingType) || ObjectChecker.areEqual(roundingType, "NONE"))
			return value;
		if (ObjectChecker.isEmptyOrZero(roundingValue))
			roundingValue = BigDecimal.ONE;
		BigDecimal priceRemainder = value.remainder(roundingValue);
		if (ObjectChecker.isEmptyOrZero(priceRemainder))
			return value;
		if (ObjectChecker.areEqual(roundingType, "CEILING"))
		{
			value = value.subtract(priceRemainder);
			value = value.add(roundingValue);
		}
		else if (ObjectChecker.areEqual(roundingType, "FLOOR"))
		{
			value = value.subtract(priceRemainder);
		}
		else if (ObjectChecker.areEqual(roundingType, "HALF_UP"))
		{
			BigDecimal roundingHalf = NaMaMath.divide(roundingValue, new BigDecimal(2), fractionalDecimalPlaces);
			if (NaMaMath.isLessThanOrEqual(roundingHalf, priceRemainder))
			{
				value = value.subtract(priceRemainder);
				value = value.add(roundingValue);
			}
			else
			{
				value = value.subtract(priceRemainder);
			}
		}
		return value;
	}

	public static BigDecimal absValue(BigDecimal value)
	{
		if (ObjectChecker.isEmptyOrZero(value))
			return BigDecimal.ZERO;
		return value.abs();
	}

	public static <T> Long totalize(List<T> list, Function<T, Long> longGetter)
	{
		return totalizeStream(list.stream().map(longGetter));
	}

	public static Long totalizeStream(Stream<Long> stream)
	{
		return stream.reduce(0L, Long::sum);
	}

	public static BigDecimal totalizeDecimalStream(Stream<BigDecimal> stream)
	{
		return stream.reduce(BigDecimal.ZERO, BigDecimal::add);
	}

	public static BigDecimal negate(BigDecimal n, Boolean isDiscount)
	{
		if (ObjectChecker.isTrue(isDiscount))
			return ObjectChecker.toZeroIfNull(n).negate();
		return n;
	}
}
