package com.namasoft.common.utils;

import com.namasoft.common.Pair;
import com.namasoft.common.constants.SettingsIDs;
import com.namasoft.common.utilities.*;
import com.namasoft.common.utils.tafqeet.CurrencyDetails;
import org.apache.commons.io.IOUtils;

import java.io.*;
import java.net.*;
import java.nio.file.*;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;

public class GeneralSettings
{
	private static List<String> currencyDetails;
	private static List<PropertyGetterInterceptor> interceptors = new CopyOnWriteArrayList<>();
	protected static Properties properties = null;
	public static String settingsFilePath;
	private static final Object lock = new Object();

	public static void addInterceptor(PropertyGetterInterceptor interceptor)
	{
		interceptors.add(interceptor);
	}

	public static void addInterceptorIfNotFound(PropertyGetterInterceptor interceptor)
	{
		if (interceptors.contains(interceptor))
			return;
		interceptors.add(interceptor);
	}

	public static void reset()
	{
		synchronized (lock)
		{
			properties = null;
			CurrencyDetails.reset();
			load();
		}
		for (PropertyGetterInterceptor interceptor : interceptors)
			interceptor.onGeneralSettingsReset();
	}

	public static Properties load()
	{
		if (properties != null)
			return properties;
		synchronized (lock)
		{
			properties = new Properties();
			try (InputStream settingStream = getInputStream();)
			{
				if (settingStream == null)
					return properties;
				properties.load(settingStream);
			}
			catch (Throwable e)
			{
				NaMaLogger.info("Could not read settings", e);
			}
		}
		return properties;
	}

	private static InputStream getInputStream() throws IOException
	{
		if (ObjectChecker.isNotEmptyOrNull(settingsFilePath))
			return Files.newInputStream(Paths.get(settingsFilePath));
		return GeneralSettings.class.getClassLoader().getResourceAsStream(SettingsIDs.SETTINGS_FILE_NAME);
	}

	private static OutputStream getOutputStream() throws IOException, URISyntaxException
	{
		if (ObjectChecker.isNotEmptyOrNull(settingsFilePath))
			return Files.newOutputStream(Paths.get(settingsFilePath));
		URL resource = GeneralSettings.class.getClassLoader().getResource(SettingsIDs.SETTINGS_FILE_NAME);
		if (resource == null)
			return null;
		return Files.newOutputStream(Paths.get(resource.toURI()));
	}
	public static String getProperty(String id, String defaultValue)
	{
		for (PropertyGetterInterceptor interceptor : interceptors)
		{
			if (interceptor.wouldIntercept(id))
				return interceptor.getProperty(id, defaultValue);
		}
		return load().getProperty(id, defaultValue);
	}

	public static String getProperty(String id)
	{
		return getProperty(id, null);
	}

	public static String getGuiServerURL()
	{
		String property = getProperty(SettingsIDs.GUI_SERVER_URL, "http://localhost:8080/erp/");
		if (ObjectChecker.isNotEmptyOrNull(property) && !property.endsWith("/"))
			property += "/";
		return property;
	}

	public static String getGuiServerURLWithoutFallBack()
	{
		return getProperty(SettingsIDs.GUI_SERVER_URL);
	}

	public static boolean isCloudeMode()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty(SettingsIDs.CLOUD_ACTIVE, "false")));
	}

	public static DBType getDBType()
	{
		String property = getProperty(SettingsIDs.DATA_BASE_TYPE, DBType.MYSQL.name());
		if (ObjectChecker.isEmptyOrNull(property))
		{
			return DBType.MYSQL;
		}
		try
		{
			return DBType.valueOf(property.toUpperCase());
		}
		catch (Exception e)
		{
			return DBType.MYSQL;
		}
	}

	public static boolean isAllowAnyCharInCode()
	{
		return true;
	}

	public static boolean isHalted()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty(SettingsIDs.IS_SYSTEM_HALTED)));
	}

	public static List<String> readCurrencyDetails()
	{
		readCurrencyDetailsIntoMemory();
		return currencyDetails;
	}

	@SuppressWarnings("unchecked")
	private static void readCurrencyDetailsIntoMemory()
	{
		if (currencyDetails == null)
		{
			try
			{
				InputStream settingStream = GeneralSettings.class.getClassLoader().getResourceAsStream(SettingsIDs.CURRENCIES_FILE_NAME);
				currencyDetails = IOUtils.readLines(settingStream, "utf-8");
				settingStream.close();
			}
			catch (Exception e)
			{
				currencyDetails = Collections.emptyList();
			}
		}
	}

	public static String getServerAddress()
	{
		if (ObjectChecker.isNotEmptyOrNull(System.getProperty("namaserver")))
			return System.getProperty("namaserver");
		String serverAddress = getProperty(SettingsIDs.SERVER_ADDRESS, "http://localhost:8080/");
		if (!serverAddress.endsWith("/"))
			serverAddress += "/";
		return serverAddress;
	}

	public static String getServerId()
	{
		return getProperty(SettingsIDs.SERVER_ID);
	}
	public static String getBusClientURL()
	{
		return getProperty(SettingsIDs.BUS_CLIENT);
	}

	public static Date getConfiguredValueDate()
	{
		return getDateFromString(getProperty(SettingsIDs.VALUE_DATE));
	}

	public static Date getConfiguredIssueDate()
	{
		return getDateFromString(getProperty(SettingsIDs.ISSUE_DATE));
	}

	private static Date getDateFromString(String valueDate)
	{
		if (ObjectChecker.isEmptyOrNull(valueDate))
			return null;
		String[] parts = valueDate.split("-");
		if (parts.length != 3)
			return null;
		Calendar calendar = Calendar.getInstance();
		calendar.set(ObjectChecker.valueOfInteger(parts[2]), ObjectChecker.valueOfInteger(parts[1]) - 1, ObjectChecker.valueOfInteger(parts[0]));
		return calendar.getTime();
	}

	public static boolean shouldAcknowledgeFailedMessages()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty(SettingsIDs.SHOULD_ACKNOWLEDGE_FAILED_MESSAGES)));
	}

	public static boolean getPreventViewingPublicDocuments()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty(SettingsIDs.PREVENT_VIEWING_PUBLIC_DOCUMENTS)));
	}

	public static boolean isHaltReplicatingSystemFiles()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty(SettingsIDs.HALT_SYSTEM_FILES_REPLICATION)));
	}

	public static String getTempFolder()
	{
		String defaultalue = PlatformInfo.isWindows() ? "c:/nama/temp" : "/var/nama/temp";
		return getProperty(SettingsIDs.TEMP_FOLDER, defaultalue);
	}

	public static String getImagesServletAddress(String relativeReportFolder)
	{
		return getGuiServerURL() + "RPTImagesServlet?src=" + relativeReportFolder;
	}

	public static boolean isSkipPrecommitInReplication()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty(SettingsIDs.SKIP_PRECOMMIT_IN_REPLICATION)));
	}

	public static String getApprovalsLink()
	{
		return getProperty(SettingsIDs.APPROVALS_SERVER_URL);
	}

	public static String getPublicDimensionId(String simpleName)
	{
		return getProperty(simpleName);
	}

	public static String getAdminUserId(String userAdmin)
	{
		return getProperty(userAdmin);
	}

	public static String getConfigGroupId(String configGroupId)
	{
		return getProperty(configGroupId);
	}

	public static String getConfigEntryId(String configEntryId)
	{
		return getProperty(configEntryId);
	}

	public static boolean isShowDetailedErrors()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty("detailederror")));
	}

	public static boolean isCostSchedulingEnabled()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty(SettingsIDs.ENABLE_COST_SCHEDULING)));
	}

	public static boolean isChangingDimensionsAllowed()
	{
		return ObjectChecker.isTrue(ObjectChecker.valueOfBoolean(getProperty(SettingsIDs.ALLOW_CHANGING_DIMENSIONS)));
	}

	public static String getCustomerName()
	{
		return getProperty(SettingsIDs.CUSTOMER);
	}

	public static String getDBName()
	{
		return getProperty(SettingsIDs.DB_NAME);
	}

	public static Properties getProperties()
	{
		return properties;
	}

	public static void setSettingsFilePath(String fileName)
	{
		GeneralSettings.settingsFilePath = fileName;
	}

	public static Boolean getPropertyBoolean(String id)
	{
		return ObjectChecker.valueOfBoolean(getProperty(id));
	}

	public static String getTomcatServiceName()
	{
		if (PlatformInfo.isWindows())
			return calcTomcatServiceNameFromPath("");
		return getProperty(SettingsIDs.TOMCAT_SERVICE);
	}

	public static String calcTomcatServiceNameFromPath(String tomcatPath)
	{
		File current = new File(tomcatPath).getAbsoluteFile();
		if (current.getName().equals("bin"))
			current = current.getParentFile();
		File bin = new File(current, "bin");
		File[] binFiles = bin.listFiles();
		if (binFiles == null)
			return "";
		File tomcatServiceExe = Arrays.stream(binFiles).filter(f -> f.getName().toLowerCase().endsWith(".exe"))
				.filter(f -> !f.getName().toLowerCase().endsWith("w.exe")).findFirst().orElse(null);
		if (tomcatServiceExe == null)
			return "";
		return tomcatServiceExe.getName().substring(0, tomcatServiceExe.getName().length() - ".exe".length());
	}

	public static void removeProperty(String property)
	{
		load().remove(property);
		saveProperties();
	}

	public static void setProperty(String name, String value)
	{
		load().setProperty(name, value);
		saveProperties();
	}

	private static void saveProperties()
	{
		try
		{
			OutputStream outputStream = getOutputStream();
			getProperties().store(outputStream, "");
			outputStream.close();
			reset();
		}
		catch (Exception e)
		{
			throw new RuntimeException(e);
		}
	}

	public static String getSubRelease()
	{
		String subRelease = getProperty("sub-release", "");
		if (subRelease.equalsIgnoreCase("gui3"))
			return "";
		return subRelease;
	}

	public static String fetchCrmNamasoftUrl()
	{
		if (ObjectChecker.isTrue(getProperty("use-namasoft-for-nlm")))
			return "https://www.namasoft.com";
		return "https://crm.namasoft.com";
	}

	public static String fetchNlmNamasoftUrl()
	{
		if (ObjectChecker.isTrue(getProperty("use-namasoft-for-nlm")))
			return "https://www.namasoft.com";
		return "https://nlm.namasoft.com";
	}

	public static boolean shouldDoNotHashPasswords()
	{
		return ObjectChecker.isTrue(getProperty(SettingsIDs.USE_LDAP_FOR_AUTHENTICATION)) || ObjectChecker.isTrue(
				getProperty(SettingsIDs.USE_CUSTOM_PASSWORD_VALIDATOR));
	}

	protected static String findSecuredPassword(String id, String type)
	{
		String dbPass = getProperty(id, "");
		if (dbPass.startsWith(CryptoUtils.prefix))
		{
			String password = dbPass.substring(CryptoUtils.prefix.length());
			return CryptoUtils.decrypt(password);
		}
		String encryptedPass = CryptoUtils.prefix.concat(CryptoUtils.encrypt(dbPass));
		try
		{
			OutputStream out = getOutputStream();
			if (out != null)
			{
				if (type.equalsIgnoreCase("DB"))
					getProperties().setProperty(SettingsIDs.DB_PASSWORD, encryptedPass);
				else if (type.equalsIgnoreCase("APIKey"))
					getProperties().setProperty(SettingsIDs.API_KEY, encryptedPass);
				else
					getProperties().setProperty(SettingsIDs.LOGIN_PASSWORD, encryptedPass);
				getProperties().store(out, "---");
				out.close();
			}
		}
		catch (Exception e)
		{
			NaMaLogger.error(e);
		}

		return dbPass;
	}

	public static void saveSetting(Properties properties, Callback<IOException> callback)
	{
		try(FileOutputStream out = new FileOutputStream(settingsFilePath);)
		{
			properties.store(out, "save setting");
		}
		catch (IOException e)
		{
			callback.done(e);
		}
	}

	public static void savePropertiesAsSections(String settingsFilePath)
	{
		List<Pair<String, List<String>>> orderedSections = new ArrayList<>();
		orderedSections.add(new Pair<>("DB Settings:",
				List.of(SettingsIDs.POS_DB_NAME, SettingsIDs.DATA_BASE_TYPE, SettingsIDs.POSDBSERVER_URL, SettingsIDs.POSDBSERVER_PORT_NUM,
						SettingsIDs.DB_USER, SettingsIDs.DB_PASSWORD)));
		orderedSections.add(new Pair<>("Server Settings:",
				List.of(SettingsIDs.POSSERVER_URL, SettingsIDs.POSSERVER_PORT_NUM, SettingsIDs.LOGIN_ID, SettingsIDs.LOGIN_PASSWORD,
						SettingsIDs.GUI_SERVER_URL)));
		orderedSections.add(new Pair<>("Read-Write Settings:",
				List.of(SettingsIDs.POSMachineCanWrite, SettingsIDs.POSMachineCanRead, SettingsIDs.POSMachineReadTime,
						SettingsIDs.POSReadRecordsCount)));
		orderedSections.add(new Pair<>("Register Settings:", List.of(SettingsIDs.REGISTERY_CODE)));
		orderedSections.add(new Pair<>("Language Settings:", List.of(SettingsIDs.DEFAULT_LANG)));
		try(FileOutputStream out = new FileOutputStream(settingsFilePath))
		{
			Set<String> handledKeys = new HashSet<>();
			StringBuilder content = new StringBuilder();
			for (Pair<String, List<String>> section : orderedSections)
			{
				content.append("#Section: ").append(section.getX()).append("\n");
				for (String key : section.getY())
				{
					appendKeyAndValToPropContent(key, content, handledKeys);
				}
			}
			if (!handledKeys.containsAll(properties.keySet()))
				content.append("#Others:\n");
			for (Object key : properties.keySet())
			{
				if (handledKeys.contains(key))
					continue;
				appendKeyAndValToPropContent(ObjectChecker.toStringOrEmpty(key), content, handledKeys);
			}
			out.write(content.toString().getBytes());
		}
		catch (IOException e)
		{
			NaMaLogger.error(e);
		}
	}

	private static void appendKeyAndValToPropContent(String key, StringBuilder content, Set<String> handledKeys)
	{
		if (properties.get(key) == null)
			return;
		content.append(saveConvert(key, true, true)).append("=").append(saveConvert(properties.getProperty(key), false, true)).append("\n");
		handledKeys.add(key);
	}

	private static String saveConvert(String theString, boolean escapeSpace, boolean escapeUnicode)
	{
		int len = theString.length();
		int bufLen = len * 2;
		if (bufLen < 0)
		{
			bufLen = Integer.MAX_VALUE;
		}
		StringBuilder outBuffer = new StringBuilder(bufLen);
		HexFormat hex = HexFormat.of().withUpperCase();
		for (int x = 0; x < len; x++)
		{
			char aChar = theString.charAt(x);
			// Handle common case first, selecting largest block that
			// avoids the specials below
			if ((aChar > 61) && (aChar < 127))
			{
				if (aChar == '\\')
				{
					outBuffer.append('\\');
					outBuffer.append('\\');
					continue;
				}
				outBuffer.append(aChar);
				continue;
			}
			switch (aChar)
			{
			case ' ':
				if (x == 0 || escapeSpace)
					outBuffer.append('\\');
				outBuffer.append(' ');
				break;
			case '\t':
				outBuffer.append('\\');
				outBuffer.append('t');
				break;
			case '\n':
				outBuffer.append('\\');
				outBuffer.append('n');
				break;
			case '\r':
				outBuffer.append('\\');
				outBuffer.append('r');
				break;
			case '\f':
				outBuffer.append('\\');
				outBuffer.append('f');
				break;
			case '=': // Fall through
			case ':': // Fall through
			case '#': // Fall through
			case '!':
				outBuffer.append('\\');
				outBuffer.append(aChar);
				break;
			default:
				if (((aChar < 0x0020) || (aChar > 0x007e)) & escapeUnicode)
				{
					outBuffer.append("\\u");
					outBuffer.append(hex.toHexDigits(aChar));
				}
				else
				{
					outBuffer.append(aChar);
				}
			}
		}
		return outBuffer.toString();
	}
}
