package com.namasoft.common.utils.ItemBarCodeParser;
import com.namasoft.common.fieldids.newids.supplychain.*;

import com.namasoft.common.constants.SCEntities;
import com.namasoft.common.flatobjects.DateFieldUtils;
import com.namasoft.common.flatobjects.GenericValue;
import com.namasoft.common.utilities.NaMaLogger;
import com.namasoft.common.utilities.NaMaMath;
import com.namasoft.common.utilities.ObjectChecker;
import com.namasoft.common.utilities.StringUtils;
import com.namasoft.common.utils.TimePeriodUtilsNormal;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

public class CommonItemBarcodeParser<RefParser extends CommonReferencePropertyValueParser>
{
	private RefParser referenceParser;

	public CommonItemBarcodeParser(RefParser parser)
	{
		this.referenceParser = parser;
	}

	private static class BigDecimalPropertyValueParser extends PropertyValueParserBase
	{
		private BigDecimalPropertyValueParser(String propertyId)
		{
			super(propertyId);
		}

		@Override
		public Object parse(String s, DTOCommonItemBarcodePart part)
		{
			return parseNumber(s, part);
		}
	}

	private static class DatePropertyValueParser extends PropertyValueParserBase
	{
		private DatePropertyValueParser(String propertyId)
		{
			super(propertyId);
		}

		@Override
		public Object parse(String s, DTOCommonItemBarcodePart part)
		{
			return parseDate(s, part);
		}
	}

	private static class IntegerPropertyValueParser extends PropertyValueParserBase
	{
		private IntegerPropertyValueParser(String propertyId)
		{
			super(propertyId);
		}

		@Override
		public Object parse(String s, DTOCommonItemBarcodePart part)
		{
			return parseInteger(s, part);
		}
	}

	private static class StringPropertyValueParser extends PropertyValueParserBase
	{
		private StringPropertyValueParser(String propertyId)
		{
			super(propertyId);
		}

		@Override
		public Object parse(String s, DTOCommonItemBarcodePart part)
		{
			return s;
		}
	}

	private static Object parseDate(String s, DTOCommonItemBarcodePart part)
	{
		if (ObjectChecker.isEmptyOrNull(s))
			return null;
		if (ObjectChecker.isNotEmptyOrNull(part.getFormat()))
		{
			try
			{
				SimpleDateFormat format = new SimpleDateFormat(part.getFormat());
				return format.parse(s);
			}
			catch (Exception e)
			{
				NaMaLogger.error(e);
			}
		}
		try
		{
			return DateFieldUtils.deserializeDate(s, TimePeriodUtilsNormal::addTimePeriodToDate);
		}
		catch (Exception e)
		{
			NaMaLogger.error(e);
		}
		return null;
	}

	private static Object parseInteger(String s, DTOCommonItemBarcodePart part)
	{
		BigDecimal parsed = parseNumber(s, part);
		if (parsed == null)
			return null;
		return parsed.intValue();
	}

	private static BigDecimal parseNumber(String s, DTOCommonItemBarcodePart part)
	{
		if (ObjectChecker.isEmptyOrNull(s))
			return null;
		if (ObjectChecker.isNotEmptyOrNull(part.getFormat()))
		{
			try
			{
				DecimalFormat format = new DecimalFormat(part.getFormat());
				format.setParseBigDecimal(true);
				BigDecimal parsed = (BigDecimal) format.parse(s);
				return applyMultiplyByAndDivideOn(part, parsed);
			}
			catch (Exception e)
			{
				NaMaLogger.error(e);
			}
		}
		try
		{
			return applyMultiplyByAndDivideOn(part, new BigDecimal(s));
		}
		catch (Exception e)
		{
			NaMaLogger.error(e);
		}
		return null;
	}

	private static BigDecimal applyMultiplyByAndDivideOn(DTOCommonItemBarcodePart part, BigDecimal parsed)
	{
		if (ObjectChecker.isNotEmptyOrZero(part.getMultiplyBy()))
			parsed = parsed.multiply(part.getMultiplyBy());
		if (ObjectChecker.isNotEmptyOrZero(part.getDivideOn()))
			parsed = NaMaMath.divide(parsed, part.getDivideOn(), 10);
		return parsed;
	}

	public String parseBarcode(final String originalItemCode, List<GenericValue> retValues, List<DTOCommonItemBarcodeSpecs> allSpecs)
	{
		String remainingItemCode = originalItemCode;
		if (ObjectChecker.isEmptyOrNull(allSpecs))
			return remainingItemCode;
		String theCode = "";
		boolean foundApplicableSpecs = false;
		DTOCommonItemBarcodeSpecs specs = null;
		for (int i = 0; i < allSpecs.size(); i++)
		{
			specs = allSpecs.get(i);
			if (ObjectChecker.isEmptyOrNull(specs) || ObjectChecker.isEmptyOrNull(specs.getParts()))
				continue;
			if (ObjectChecker.isNotEmptyOrZero(specs.getMinCodeLength()) && originalItemCode.length() < specs.getMinCodeLength())
				continue;
			if (ObjectChecker.isNotEmptyOrZero(specs.getMaxCodeLength()) && originalItemCode.length() > specs.getMaxCodeLength())
				continue;
			if (ObjectChecker.isNotEmptyOrNull(specs.getPrefix()))
			{
				if (!remainingItemCode.startsWith(specs.getPrefix()))
					continue;
				if (ObjectChecker.isFalseOrNull(specs.getTreatPrefixAsPartOfFirstProperty()))
					remainingItemCode = remainingItemCode.substring(specs.getPrefix().length());
			}
			foundApplicableSpecs = true;
			break;
		}
		if (!foundApplicableSpecs)
			return originalItemCode;
		List<GenericValue> values = new ArrayList<>();
		for (DTOCommonItemBarcodePart part : specs.getParts())
		{
			if (ObjectChecker.isEmptyOrNull(remainingItemCode))
				break;
			String current = "";
			if (ObjectChecker.isNotEmptyOrZero(part.getLength()) && remainingItemCode.length() >= part.getLength())
			{
				current = remainingItemCode.substring(0, part.getLength());
				remainingItemCode = remainingItemCode.substring(part.getLength());
			}
			else if (ObjectChecker.isNotEmptyOrNull(part.getSeparator()))
			{
				current = StringUtils.substringBefore(remainingItemCode, part.getSeparator());
				remainingItemCode = StringUtils.substringAfter(remainingItemCode, part.getSeparator());
			}
			else
			{
				current = remainingItemCode;
				remainingItemCode = "";
			}
			if (ObjectChecker.areEqual(part.getProperty(), "Code"))
			{
				theCode = current;
			}
			else
			{
				PropertyValueParserBase parser = createParser(part.getProperty());
				values.add(new GenericValue(parser.propertyId(), parser.parse(current, part)));
			}
		}
		if (ObjectChecker.isEmptyOrNull(theCode))
		{
			NaMaLogger.error("Error in parsing barcode {0}, could not find theCode. ", originalItemCode);
			return originalItemCode;
		}
		retValues.addAll(values);
		return theCode;
	}

	private PropertyValueParserBase createParser(String propertyName)
	{
		switch (propertyName)
		{
		case "Box":
			return new StringPropertyValueParser(IdsOfBasicSCDocument.details_specificDimensions_box);
		case "Code":
			return new StringPropertyValueParser(IdsOfBasicSCDocument.details_item_itemCode);
		case "Color":
			return new StringPropertyValueParser(IdsOfBasicSCDocument.details_specificDimensions_color);
		case "Revision":
			return new StringPropertyValueParser(IdsOfBasicSCDocument.details_specificDimensions_revisionId);
		case "Size":
			return new StringPropertyValueParser(IdsOfBasicSCDocument.details_specificDimensions_size);
		case "Lot":
			return new StringPropertyValueParser(IdsOfBasicSCDocument.details_specificDimensions_lotId);
		case "ExpiryDate":
			return new DatePropertyValueParser(IdsOfBasicSCDocument.details_expiryDate);
		case "ProductionDate":
			return new DatePropertyValueParser(IdsOfBasicSCDocument.details_productionDate);
		case "FirstSerialNumber":
			return new StringPropertyValueParser(IdsOfBasicSCDocument.details_specificDimensions_serialNumber);
		case "SecondSerialNumber":
			return new StringPropertyValueParser(IdsOfBasicSCDocument.details_specificDimensions_secondSerial);
		case "Count":
			return new BigDecimalPropertyValueParser(IdsOfBasicSCDocument.details_quantity_measureQty);
		case "Length":
			return new BigDecimalPropertyValueParser(IdsOfBasicSCDocument.details_quantity_measures_length);
		case "Width":
			return new BigDecimalPropertyValueParser(IdsOfBasicSCDocument.details_quantity_measures_width);
		case "Height":
			return new BigDecimalPropertyValueParser(IdsOfBasicSCDocument.details_quantity_measures_height);
		case "Quantity":
			return new BigDecimalPropertyValueParser(IdsOfBasicSCDocument.details_quantity_quantity_primeQty_value);
		case "TotalPrice":
			return new BigDecimalPropertyValueParser(IdsOfSalesInvoice.details_price_price);
		case "UnitPrice":
			return new BigDecimalPropertyValueParser(IdsOfSalesInvoice.details_price_unitPrice);
		case "UOM":
			return referenceParser.instance(SCEntities.UOM, IdsOfBasicSCDocument.details_quantity_quantity_primeQty_uom);
		}
		return null;
	}
}
