package com.namasoft.common.utils;

import com.namasoft.common.constants.*;
import com.namasoft.common.utilities.*;

import java.math.BigDecimal;
import java.util.*;

public class TimePeriodUtils
{
	public static Integer diffInMonthsBetween2Dates(Date startDate, Date endDate)
	{
		return (endDate.getYear() - startDate.getYear()) * 12 + (endDate.getMonth() + 1 - startDate.getMonth());

	}

	public static Date subtractTimePeriodToDate(Date date, String uom, BigDecimal value)
	{
		if (ObjectChecker.isEmptyOrNull(uom) || ObjectChecker.isEmptyOrZero(value))
			return date;
		return subtractTimePeriodToDate(date, TimePeriodType.valueOf(uom), value);
	}

	public static Date subtractTimePeriodToDate(Date date, TimePeriodType uom, Integer value)
	{
		return addTimePeriodToDate(date, uom, BigDecimal.valueOf(value), -1, true);
	}

	public static Date subtractTimePeriodToDate(Date date, TimePeriodType uom, Long value)
	{
		return addTimePeriodToDate(date, uom, BigDecimal.valueOf(value), -1, true);
	}

	public static Date subtractTimePeriodToDate(Date date, TimePeriodType uom, BigDecimal value)
	{
		return addTimePeriodToDate(date, uom, value, -1, true);
	}

	public static Date addTimePeriodToDate(Date date, TimePeriodType uom, Integer value)
	{
		return addTimePeriodToDate(date, uom, BigDecimal.valueOf(value), 1, true);
	}

	public static Date addTimePeriodToDate(Date date, TimePeriodType uom, Long value)
	{
		return addTimePeriodToDate(date, uom, BigDecimal.valueOf(value), 1, true);
	}

	public static Date addTimePeriodToDate(Date date, TimePeriodType uom, BigDecimal value)
	{
		return addTimePeriodToDate(date, uom, value, 1, true);
	}

	static Date addTimePeriodToDate(Date date, TimePeriodType uom, BigDecimal value, int factorBy, boolean moveADay)
	{
		if (ObjectChecker.isEmptyOrZero(value) || uom == null)
			return date;
		Date newDate = (Date) date.clone();
		Calendar calendar = new GregorianCalendar();
		calendar.setTime(newDate);
		calendar.add(field(uom), value.multiply(BigDecimal.valueOf(factorBy)).intValue());
		if (!moveADay)
			return calendar.getTime();
		if (ObjectChecker.isNotEmptyOrNull(uom))
		{
			switch (uom)
			{
			case Hour:
			case Minute:
			case Second:
				break;
			case Day:
			case Week:
			case Month:
			case Year:
				calendar.add(Calendar.DAY_OF_YEAR, -1 * factorBy);
				break;
			default:
				break;
			}
		}
		return calendar.getTime();
	}

	public final static List<TimePeriodType> orderedPeriods = Arrays.asList(TimePeriodType.Year, TimePeriodType.Month, TimePeriodType.Week,
			TimePeriodType.Day, TimePeriodType.Hour);

	private static int field(TimePeriodType uom)
	{
		switch (uom)
		{
		case Day:
			return Calendar.DAY_OF_YEAR;
		case Hour:
			return Calendar.HOUR_OF_DAY;
		case Minute:
			return Calendar.MINUTE;
		case Month:
			return Calendar.MONTH;
		case Second:
			return Calendar.SECOND;
		case Week:
			return Calendar.WEEK_OF_YEAR;
		case Year:
			return Calendar.YEAR;
		}
		throw new IllegalArgumentException("Unkown time period type: " + uom);
	}

	public static SimpleEntry<BigDecimal, String> calcPeriod(Date start, Date end)
	{
		long diff = end.getTime() - start.getTime();
		return calcPeriodFromMs(diff);
	}

	public static SimpleEntry<BigDecimal, String> calcPeriodFromMs(long diff)
	{
		for (int i = 0; i < orderedPeriods.size(); i++)
		{
			if (orderedPeriods.get(i).inMs() <= diff)
			{
				if (i + 1 < orderedPeriods.size())
				{
					BigDecimal periodValue = NaMaMath.divide(BigDecimal.valueOf(diff), BigDecimal.valueOf(orderedPeriods.get(i).inMs()),
							CommonConstants.PERCENTAGE_SACLE);
					long remaining = diff % orderedPeriods.get(i + 1).inMs();
					if (remaining == 0)
						return new SimpleEntry<BigDecimal, String>((periodValue), orderedPeriods.get(i).name());
				}
			}
		}
		BigDecimal periodValue = NaMaMath.divide(BigDecimal.valueOf(diff), BigDecimal.valueOf(TimePeriodType.Hour.inMs()),
				CommonConstants.PERCENTAGE_SACLE);
		return new SimpleEntry<BigDecimal, String>((periodValue), TimePeriodType.Hour.name());
	}

	public static long guessPeriodInMS(Date start, Date end)
	{
		return end.getTime() - start.getTime();
	}

	public static SimpleEntry<BigDecimal, String> guessPeriod(Date start, Date end)
	{
		long diff = end.getTime() - start.getTime();
		for (int i = 0; i < orderedPeriods.size(); i++)
		{
			if (orderedPeriods.get(i).inMs() <= diff)
			{
				if (i + 1 < orderedPeriods.size())
				{
					long remaining = diff % orderedPeriods.get(i + 1).inMs();
					if (remaining == 0)
						return new SimpleEntry<BigDecimal, String>(new BigDecimal((double) diff / orderedPeriods.get(i).inMs()).setScale(2,
								BigDecimal.ROUND_HALF_UP), orderedPeriods.get(i).name());
				}
			}
		}
		return new SimpleEntry<BigDecimal, String>(new BigDecimal((double) diff / TimePeriodType.Hour.inMs()).setScale(2, BigDecimal.ROUND_HALF_UP),
				TimePeriodType.Hour.name());
	}

	public static SimpleEntry<Integer, String> guessPeriod(GregorianCalendar start, GregorianCalendar end)
	{
		long diff = end.getTimeInMillis() - start.getTimeInMillis();
		for (int i = 0; i < orderedPeriods.size(); i++)
		{
			if (orderedPeriods.get(i).inMs() <= diff)
			{
				if (i + 1 < orderedPeriods.size())
				{
					long remaining = diff % orderedPeriods.get(i + 1).inMs();
					if (remaining == 0)
						return new SimpleEntry<Integer, String>((int) (diff / orderedPeriods.get(i).inMs()), orderedPeriods.get(i).name());
				}
			}
		}
		return new SimpleEntry<Integer, String>((int) (diff / TimePeriodType.Hour.inMs()), TimePeriodType.Hour.name());
	}

	public static boolean isDateBetween(Date date, Date start, Date end)
	{
		return date.getTime() >= start.getTime() && date.getTime() <= end.getTime();
	}
}
