package com.namasoft.common.utils.dm.pojo;

import com.namasoft.common.NaMaDTO;
import com.namasoft.common.constants.*;
import com.namasoft.common.fieldids.CommonFieldIds;
import com.namasoft.common.layout.metadata.FieldType;
import com.namasoft.common.utilities.*;
import com.namasoft.common.utils.translation.TranslationUtil;
import jakarta.xml.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

@SuppressWarnings("serial")
@XmlAccessorType(XmlAccessType.PROPERTY)
@XmlRootElement
public class DMEntity extends NaMaDTO implements DMContainer, Comparable<DMEntity>
{
	/***
	 * Entity,DetailLine,SystemTable
	 */
	private String tableType;
	private String className;
	private String entityName;
	private String pluralArabicName;
	private String pluralEnglishName;
	private String tableName;
	private String arabicName;
	private String englishName;
	private String fullName;
	private List<DMProperty> properties = new ArrayList<>();
	private List<DMDetail> details = new ArrayList<>();
	private transient Boolean isLoaded;
	private transient Class actualClass;
	private Boolean isDetail;
	private String detailParentEntity;
	private String detailFullName;
	private Boolean posClass;
	private Boolean meCloned;

	/***
	 * WARNING: works only in domain base
	 * @return
	 */
	public boolean checkIfClassIsLoaded()
	{
		if (isLoaded == null)
		{
			try
			{
				actualClass = Class.forName(this.className);
				isLoaded = true;
			}
			catch (ClassNotFoundException e)
			{
				isLoaded = false;
			}
		}
		return isLoaded;
	}

	/***
	 * WARNING: works only in domain base
	 * @return
	 */
	public Class fetchActualClass()
	{
		if (isLoaded == null)
			checkIfClassIsLoaded();
		return actualClass;
	}

	public String getEntityName()
	{
		return entityName;
	}

	public void setEntityName(String entityName)
	{
		this.entityName = entityName;
	}

	public String getTableName()
	{
		return tableName;
	}

	public void setTableName(String tableName)
	{
		this.tableName = tableName;
	}

	public String getArabicName()
	{
		return arabicName;
	}

	public void setArabicName(String arabicName)
	{
		this.arabicName = arabicName;
	}

	public String getEnglishName()
	{
		return englishName;
	}

	public void setEnglishName(String englishName)
	{
		this.englishName = englishName;
	}

	public List<DMProperty> getProperties()
	{
		return properties;
	}

	public void setProperties(List<DMProperty> properties)
	{
		this.properties = properties;
	}

	public List<DMDetail> getDetails()
	{
		return details;
	}

	public void setDetails(List<DMDetail> details)
	{
		this.details = details;
	}

	public String getPluralArabicName()
	{
		return pluralArabicName;
	}

	public void setPluralArabicName(String pluralArabicName)
	{
		this.pluralArabicName = pluralArabicName;
	}

	public String getPluralEnglishName()
	{
		return pluralEnglishName;
	}

	public void setPluralEnglishName(String pluralEnglishName)
	{
		this.pluralEnglishName = pluralEnglishName;
	}

	public void finishDefinition()
	{
		Collections.sort(getProperties());
		Collections.sort(getDetails());
		for (DMDetail detail : getDetails())
		{
			detail.finishDefinition();
		}
	}

	@Override
	public int compareTo(DMEntity o)
	{
		if (o == null)
			return 1;
		return getEntityName().compareTo(o.getEntityName());
	}

	public String getClassName()
	{
		return className;
	}

	public String getTableType()
	{
		return tableType;
	}

	public void setTableType(String tableType)
	{
		this.tableType = tableType;
	}

	@Override
	public DMEntity prefixWith(String prefixWith)
	{
		if (ObjectChecker.isEmptyOrNull(prefixWith))
			return this;
		return cloneToADifferentFullName(prefixWith);
	}

	private DMEntity cloneToADifferentFullName(String fullName)
	{
		DMEntity d = new DMEntity();
		d.className = className;
		d.entityName = entityName;
		d.pluralArabicName = pluralArabicName;
		d.pluralEnglishName = pluralEnglishName;
		d.tableName = tableName;
		d.arabicName = arabicName;
		d.englishName = englishName;
		d.isDetail = isDetail;
		d.details = new ArrayList<>(details);
		d.properties = new ArrayList<>(properties);
		d.tableType = tableType;
		d.detailFullName = detailFullName;
		d.detailParentEntity = detailParentEntity;
		d.posClass = posClass;
		d.meCloned = meCloned;
		d.fullName = fullName;
		return d;
	}

	public String fetchFullName()
	{
		return ObjectChecker.getFirstNotEmptyObj(fullName, "MainEntity");
	}

	@Override
	public Boolean isPosClass()
	{
		return posClass;
	}

	public void setClassName(String className)
	{
		this.className = className;
	}

	public DMPropWithParent findField(String fieldId, DMPersister persister, String prefixWith)
	{
		return findField(fieldId, persister, prefixWith, false);
	}

	public DMPropWithParent findField(String fieldId, DMPersister persister, String prefixWith, boolean translateIfNeeded)
	{
		List<DMProperty> parentsPath = new ArrayList<>();
		DMPropWithParent field = findField(fieldId, persister, prefixWith, translateIfNeeded, parentsPath);
		if (field != null)
			field.setParentsPath(parentsPath.stream().map(DMProperty::getFullName).collect(Collectors.toList()));
		return field;
	}

	public DMPropWithParent findField(String fieldId, DMPersister persister, String prefixWith, boolean translateIfNeeded,
			List<DMProperty> parentsPath)
	{
		DMProperty dmProperty = find(fieldId, getProperties(), persister);
		if (dmProperty != null)
		{
			parentsPath.add(dmProperty);
			return new DMPropWithParent(this, dmProperty, prefixWith).addSingleJoinIfNeeded();
		}
		DMDetail dmDetail = findFieldDetail(fieldId, getDetails());
		if (dmDetail == null)
		{
			return recursivelySearchForField(this, this, fieldId, persister, prefixWith, translateIfNeeded, parentsPath);
		}

		dmProperty = find(fieldId, dmDetail.getProperties(), persister);
		if (dmProperty == null)
		{
			String subFieldId = fieldId.substring(dmDetail.getFullName().length() + 1);
			if (subFieldId.contains("."))
			{
				return recursivelySearchForField(dmDetail, this, fieldId, persister, prefixWith, translateIfNeeded, parentsPath);
			}
			return null;
		}
		parentsPath.add(dmProperty);
		DMPropWithParent prop = new DMPropWithParent(dmDetail, dmProperty, prefixWith);
		prop.setRoot(this);
		return prop.addJoinForDetail(this);
	}

	private static DMProperty find(String fieldId, List<DMProperty> properties, DMPersister persister)
	{
		String idBeforeHash = fieldId;
		String genRefType = null;
		if (findLastFieldId(fieldId).contains("#"))
		{
			idBeforeHash = StringUtils.substringBeforeLast(fieldId, "#");
			genRefType = StringUtils.substringAfterLast(fieldId, "#");
		}
		DMProperty property = properties.stream().filter(ObjectChecker.equalsPredicate(idBeforeHash, DMProperty::getFullName)).findFirst()
				.orElse(null);
		if (property == null)
			return null;
		if (ObjectChecker.isNotEmptyOrNull(genRefType))
			property = property.changeTypeToReferenceAndAppendTypeToFullName(genRefType, persister);
		return property;
	}

	private static String findLastFieldId(String fieldId)
	{
		if (!fieldId.contains("{"))
			return StringUtils.substringAfterLastOrStrIfSeparatorNotFound(fieldId, ".");
		String replacement = "@@++---!!";
		for (String str : StringUtils.getAllInnerStrings(fieldId, "{", "}"))
		{
			if (!str.contains("."))
				continue;
			String strWithoutDot = str.replace(".", replacement);
			fieldId = fieldId.replace(str, strWithoutDot);
		}
		return StringUtils.substringAfterLastOrStrIfSeparatorNotFound(fieldId, ".").replace(replacement, ".");
	}
	public static String extractManualJoinEntity(String s)
	{
		return StringUtils.substringBefore(StringUtils.substringAfterLastOrStrIfSeparatorNotFound(s, "#"), "{").trim();
	}

	private static DMDetail findFieldDetail(String fieldId, List<DMDetail> details)
	{
		return details.stream().filter(d -> fieldId.startsWith(d.getFullName() + ".")).findFirst().orElse(null);
	}

	private static DMPropWithParent recursivelySearchForField(DMContainer container, DMEntity root, String fullId, DMPersister persister,
			String prefixWith, boolean translateIfNeeded, List<DMProperty> parentsPath)
	{
		if (!fullId.contains("."))
			return null;
		String[] parts = fullId.split("\\.");

		for (int i = parts.length - 1; i >= 0; i--)
		{
			String leftHandSidePart = Arrays.asList(parts).subList(0, i).stream().collect(Collectors.joining("."));
			String rightHandSidePart = Arrays.asList(parts).subList(i, parts.length).stream().collect(Collectors.joining("."));

			DMProperty property = find(leftHandSidePart, container.getProperties(), persister);
			if (property == null)
				continue;
			parentsPath.add(property);
			property = property.prefixWith(prefixWith);
			List<DMPropWithParent.DMJoin> joins = new ArrayList<>();
			if (ObjectChecker.isNotEmptyOrNull(property.getReferenceTo()))
			{
				addJoinsForRefMidProperty(container, root, prefixWith, property, joins);
				DMPropWithParent prop = findDMPropertyInReferencedEntity(fullId, persister, rightHandSidePart, property, prefixWith,
						translateIfNeeded, parentsPath);
				if (prop == null)
					return null;
				prop.getJoins().addAll(0, joins);
				prop.assignParentPropertyIfEmpty(property);
				return prop;
			}
			else if (ObjectChecker.areEqual(property.getFieldType(), FieldType.Genericreference))
			{
				//subsidiaryActualCode, subsidiaryCode, subsidiaryEntityType, subsidiaryId
				List<String> genRefSubFields = Arrays.asList(CommonFieldIds.ACTUAL_CODE, CommonFieldIds.CODE, CommonFieldIds.ENTITY_TYPE,
						CommonFieldIds.ID);
				int subPropIndex = genRefSubFields.indexOf(rightHandSidePart);
				if (subPropIndex > 0)
				{
					String columnName = StringUtils.csvLineToList(property.getColumnName()).get(subPropIndex).trim();
					DMProperty prop = persister.findEntity(DomainBaseEntities.LegalEntity).findField(rightHandSidePart, persister, null)
							.getProperty();
					prop = prop.cloneData();
					prop.setFullName(leftHandSidePart + "." + rightHandSidePart);
					prop.setColumnName(columnName);
					parentsPath.add(prop);
					return new DMPropWithParent(container, prop, prefixWith).addSingleJoinIfNeeded();
				}
			}
			return null;
		}
		return null;
	}

	private static void addJoinsForRefMidProperty(DMContainer container, DMEntity root, String prefixWith, DMProperty property,
			List<DMPropWithParent.DMJoin> joins)
	{
		String mainTableAlias = StringUtils.toCamelCaseAndReplaceDotsWith_(root.fetchFullName());
		if (ObjectChecker.isNotEmptyOrNull(prefixWith))
			mainTableAlias = StringUtils.toCamelCaseAndReplaceDotsWith_(prefixWith);
		if (container instanceof DMEntity)
		{
			joins.add(DMPropWithParent.createToOneJoin(root.getTableName(), mainTableAlias,
					ObjectChecker.getFirstNotEmptyObj(prefixWith, root.fetchFullName()), property));
		}
		else
		{
			joins.add(DMPropWithParent.createToManyJoin(root.getTableName(), mainTableAlias,
					ObjectChecker.getFirstNotEmptyObj(prefixWith, root.fetchFullName()), ((DMDetail) container).prefixWith(prefixWith)));
			joins.add(DMPropWithParent.createToOneJoin(CollectionsUtility.getLast(joins).getJoinedTableName(),
					CollectionsUtility.getLast(joins).getJoinedTableAlias(), CollectionsUtility.getLast(joins).getJoinedTablePropertyName(),
					property));
		}
	}

	private static DMPropWithParent findDMPropertyInReferencedEntity(String fullId, DMPersister persister, String rightHandSidePart,
			DMProperty property, String prefixWith, boolean translateIfNeeded, List<DMProperty> parentsPath)
	{
		String leftHandSide = fullId.substring(0, fullId.length() - rightHandSidePart.length() - 1);
		if (ObjectChecker.isNotEmptyOrNull(prefixWith))
			leftHandSide = prefixWith + "." + leftHandSide;
		DMPropWithParent field = persister.findField(property.getReferenceTo(), rightHandSidePart, leftHandSide, translateIfNeeded, parentsPath);
		if (field == null)
			return null;
		return field;
	}

	public void translate()
	{
		translateProperties(getProperties());
		setArabicName(translate(Language.Arabic, getEntityName()));
		setPluralArabicName(translate(Language.Arabic, getEntityName() + "s"));
		setEnglishName(translate(Language.English, getEntityName()));
		setPluralEnglishName(translate(Language.English, getEntityName() + "s"));
		for (DMDetail detail : getDetails())
		{
			detail.setArabicName(translate(Language.Arabic, getEntityName() + "." + detail.getFullName()));
			detail.setEnglishName(translate(Language.English, getEntityName() + "." + detail.getFullName()));
			translateProperties(detail.getProperties());
		}
	}

	private void translateProperties(List<DMProperty> props)
	{
		if (props == null)
			return;
		for (DMProperty property : props)
		{
			property.setArabicName(translate(Language.Arabic, getEntityName() + "." + property.getFullName()));
			property.setEnglishName(translate(Language.English, getEntityName() + "." + property.getFullName()));
		}
	}

	private static String translate(Language lang, String str)
	{
		String translation = TranslationUtil.translate(lang, str);
		if (ObjectChecker.areEqual(str, translation))
			return "";
		return translation;
	}

	public void setIsDetail(Boolean isDetail)
	{
		this.isDetail = isDetail;
	}

	public Boolean getIsDetail()
	{
		return isDetail;
	}

	public void setDetailParentEntity(String detailParentEntity)
	{
		this.detailParentEntity = detailParentEntity;
	}

	public String getDetailParentEntity()
	{
		return detailParentEntity;
	}

	public void setDetailFullName(String detailFullName)
	{
		this.detailFullName = detailFullName;
	}

	public String getDetailFullName()
	{
		return detailFullName;
	}

	public DMProperty findPropertyByColumnName(String column)
	{
		return getProperties().stream().filter(f -> ObjectChecker.areEqual(f.getColumnName(), column)).findFirst().orElse(null);
	}

	public Boolean getPosClass()
	{
		return posClass;
	}

	public void setPosClass(Boolean posClass)
	{
		this.posClass = posClass;
	}

	public DMEntity cloneData()
	{
		DMEntity c = new DMEntity();
		c.setIsDetail(getIsDetail());
		c.setDetails(getDetails());
		c.setProperties(new ArrayList<>(getProperties()));
		c.setTableName(getTableName());
		c.setEntityName(getEntityName());
		c.setClassName(getClassName());
		c.setTableType(getTableType());
		c.setEnglishName(getEnglishName());
		c.setArabicName(getArabicName());
		c.setDetailFullName(getDetailFullName());
		c.setDetailParentEntity(getDetailParentEntity());
		c.setPluralArabicName(getPluralArabicName());
		c.setPluralEnglishName(getPluralEnglishName());
		c.setPosClass(getPosClass());
		c.setMeCloned(true);
		return c;
	}

	public Boolean getMeCloned()
	{
		return meCloned;
	}

	public void setMeCloned(Boolean meCloned)
	{
		this.meCloned = meCloned;
	}

	public void translateIfNeeded()
	{
		if (ObjectChecker.isAnyNotEmptyOrNull(getArabicName(), getEnglishName()))
			return;
		translate();
	}
}
