package com.namasoft.common.utils.tafqeet;

import com.namasoft.common.constants.Language;
import com.namasoft.common.utilities.NaMaMath;
import com.namasoft.common.utilities.ObjectChecker;
import com.namasoft.common.utils.NaMaLayersConnector;

import java.math.BigDecimal;

public class Converter
{

	private static final String ENGLISH_SUFFIX_TEXT = "only.";
	private static final String FRENCH_SUFFIX_TEXT = "seulement";
	private static final String ARABIC_PREFIX_TEXT = "فقط";
	private static final String ARABIC_SUFFIX_TEXT = "لاغير.";

	private static final String[] englishOnes = new String[] { "Zero", "One", "Two", "Three", "Four", "Five", "Six", "Seven", "Eight", "Nine", "Ten",
			"Eleven", "Twelve", "Thirteen", "Fourteen", "Fifteen", "Sixteen", "Seventeen", "Eighteen", "Nineteen" };

	private static String[] englishTens = new String[] { "Twenty", "Thirty", "Forty", "Fifty", "Sixty", "Seventy", "Eighty", "Ninety" };

	private static String[] englishGroup = new String[] { "Hundred", "Thousand", "Million", "Billion", "Trillion", "Quadrillion", "Quintillion",
			"Sextillian", "Septillion", "Octillion", "Nonillion", "Decillion", "Undecillion", "Duodecillion", "Tredecillion", "Quattuordecillion",
			"Quindecillion", "Sexdecillion", "Septendecillion", "Octodecillion", "Novemdecillion", "Vigintillion", "Unvigintillion",
			"Duovigintillion", "10^72", "10^75", "10^78", "10^81", "10^84", "10^87", "Vigintinonillion", "10^93", "10^96", "Duotrigintillion",
			"Trestrigintillion" };

	private static String[] arabicOnes = new String[] { "", "واحد", "اثنان", "ثلاثة", "أربعة", "خمسة", "ستة", "سبعة", "ثمانية", "تسعة", "عشرة",
			"أحد عشر", "اثنا عشر", "ثلاثة عشر", "أربعة عشر", "خمسة عشر", "ستة عشر", "سبعة عشر", "ثمانية عشر", "تسعة عشر" };

	private static String[] arabicFeminineOnes = new String[] { "", "إحدى", "اثنتان", "ثلاث", "أربع", "خمس", "ست", "سبع", "ثمان", "تسع", "عشر",
			"إحدى عشرة", "اثنتا عشرة", "ثلاث عشرة", "أربع عشرة", "خمس عشرة", "ست عشرة", "سبع عشرة", "ثماني عشرة", "تسع عشرة" };

	private static String[] arabicTens = new String[] { "عشرون", "ثلاثون", "أربعون", "خمسون", "ستون", "سبعون", "ثمانون", "تسعون" };

	private static String[] arabicHundreds = new String[] { "", "مائة", "مئتان", "ثلاثمائة", "أربعمائة", "خمسمائة", "ستمائة", "سبعمائة", "ثمانمائة",
			"تسعمائة" };

	private static String[] arabicTwos = new String[] { "مئتان", "ألفان", "مليونان", "ملياران", "تريليونان", "كوادريليونان", "كوينتليونان",
			"سكستيليونان" };
	private static String[] arabicAppendedTwos = arabicTwos;//Banks in Egypt do not accept removing the last نون
	//new String[] { "مئتا", "ألفا", "مليونا", "مليارا", "تريليونا", "كوادريليونا", "كوينتليونا",	"سكستيليونا" };
	private static String[] arabicGroup = new String[] { "مائة", "ألف", "مليون", "مليار", "تريليون", "كوادريليون", "كوينتليون", "سكستيليون" };
	private static String[] arabicAppendedGroup = new String[] { "", "ألفاً", "مليوناً", "ملياراً", "تريليوناً", "كوادريليوناً", "كوينتليوناً",
			"سكستيليوناً" };
	private static String[] arabicPluralGroups = new String[] { "", "آلاف", "ملايين", "مليارات", "تريليونات", "كوادريليونات", "كوينتليونات",
			"سكستيليونات" };

	public static String convertToWords(BigDecimal value, String currencyCode, Language language)
	{

		CurrencyDetails currencyDetails = CurrencyDetails.get(currencyCode);
		BigDecimal number = ObjectChecker.toZeroIfNull(NaMaMath.round(value, currencyDetails.getPartPrecision())).abs();

		if (language == Language.English)
		{
			if (NaMaLayersConnector.getInstance().usingFrench())
				return convertToFrench(number, currencyDetails);
			else
				return convertToEnglish(number, currencyDetails);
		}
		else
		{
			return convertToArabic(number, currencyDetails);
		}

	}

	public static void main(String[] args)
	{

	}

	private static String convertToEnglish(BigDecimal number, CurrencyDetails currencyDetails)
	{

		String[] splits = number.toString().split("\\.");
		long intergerValue = getIntegerPart(splits);
		int decimalValue = getDecimalPart(currencyDetails, splits);

		if (number.compareTo(new BigDecimal(0)) == 0)
			return "Zero";

		String decimalString = convertDecimalPartToEnWords(decimalValue);
		String retVal = convertIntegerPartToEnWords(new BigDecimal(intergerValue));

		String formattedNumber = "";
		formattedNumber += (retVal != "") ? retVal : "";
		formattedNumber += (retVal != "") ? (intergerValue == 1 ? currencyDetails.getEnglishCurrencyName() : currencyDetails
				.getEnglishPluralCurrencyName()) : "";
		formattedNumber += (decimalString != "") ? " and " : "";
		formattedNumber += (decimalString != "") ? decimalString : "";
		formattedNumber += (decimalString != "") ? " "
				+ (decimalValue == 1 ? currencyDetails.getEnglishCurrencyPartName() : currencyDetails.getEnglishPluralCurrencyPartName()) : "";
		formattedNumber += String.format(" %s", ENGLISH_SUFFIX_TEXT);

		return formattedNumber;
	}

	public static String convertToFrench(BigDecimal value, String currencyCode)
	{
		CurrencyDetails currencyDetails = CurrencyDetails.get(currencyCode);
		BigDecimal number = NaMaMath.round(value, currencyDetails.getPartPrecision());
		return convertToFrench(number, currencyDetails);
	}

	public static String convertToFrench(BigDecimal number, CurrencyDetails currencyDetails)
	{
		String[] splits = number.toString().split("\\.");
		long intergerValue = getIntegerPart(splits);
		int decimalValue = getDecimalPart(currencyDetails, splits);

		String decimalString = FrenchNumberToWords.convert(decimalValue);
		String retVal = FrenchNumberToWords.convert(intergerValue);

		StringBuilder formattedNumber = new StringBuilder("");
		formattedNumber.append(retVal).append(" ");
		formattedNumber.append(!retVal.equals("") ?
				(intergerValue == 1 ? currencyDetails.getEnglishCurrencyName() : currencyDetails.getEnglishPluralCurrencyName()) :
				"");
		formattedNumber.append(!decimalString.equals("") ? " et " : "");
		formattedNumber.append(decimalString);
		formattedNumber.append(!decimalString.equals("") ?
				" " + (decimalValue == 1 ? currencyDetails.getEnglishCurrencyPartName() : currencyDetails.getEnglishPluralCurrencyPartName()) + " " :
				"");
		formattedNumber.append(FRENCH_SUFFIX_TEXT);
		return formattedNumber.toString();
	}

	private static String convertIntegerPartToEnWords(BigDecimal number)
	{
		String retVal = "";
		int group = 0;

		if (number.compareTo(new BigDecimal(0)) < 1)
			retVal = englishOnes[0];

		else
		{
			while (number.compareTo(new BigDecimal(0)) > 0)
			{
				int numberToProcess = number.remainder(new BigDecimal(1000)).intValue();
				number = number.divideToIntegralValue(new BigDecimal(1000));
				String groupDescription = convertDecimalPartToEnWords(numberToProcess);
				if (groupDescription != "")
				{
					if (group > 0)
						retVal = String.format("%s %s", englishGroup[group], retVal);

					retVal = String.format("%s %s", groupDescription, retVal);
				}
				group++;
			}
		}
		return retVal;
	}

	private static int getDecimalPart(CurrencyDetails currencyDetails, String[] splits)
	{
		int _decimalValue;
		if (splits.length > 1)
			_decimalValue = Integer.valueOf(getDecimalValue(splits[1], currencyDetails));
		else
			_decimalValue = 0;
		return _decimalValue;
	}

	private static long getIntegerPart(String[] splits)
	{
		return Long.valueOf(splits[0]).longValue();
	}

	private static String convertToArabic(BigDecimal number, CurrencyDetails currencyDetails)
	{
		String[] splits = number.toString().split("\\.");
		Long _intergerValue = getIntegerPart(splits);
		int _decimalValue = getDecimalPart(currencyDetails, splits);
		BigDecimal tempNumber = number;

		if (tempNumber.compareTo(new BigDecimal(0)) == 0)
			return "صفر";

		// Get Text for the decimal part
		String decimalString = convertPartToArabicWords(_decimalValue, -1, new BigDecimal(0), currencyDetails, _decimalValue);
		String retVal = "";
		Byte group = 0;
		while (tempNumber.compareTo(new BigDecimal(0)) > 0)
		{
			// seperate number into groups
			int numberToProcess = tempNumber.remainder(new BigDecimal(1000)).intValue();
			tempNumber = tempNumber.divideToIntegralValue(new BigDecimal(1000));
			// convert group into its text
			String groupDescription = convertPartToArabicWords(numberToProcess, group, new BigDecimal(Math.floor(tempNumber.doubleValue())),
					currencyDetails, _intergerValue);

			if (groupDescription != "")
			{
				// here we add the new converted group to the previous
				// concatenated text
				if (group > 0)
				{
					if (retVal != "")
						retVal = String.format("%s %s", "و", retVal);

					if (numberToProcess != 2)
						if (numberToProcess % 100 != 1)
						{
							if (numberToProcess >= 3 && numberToProcess <= 10)
								// for numbers between 3 and 9 we use plural
								// name
								retVal = String.format("%s %s", arabicPluralGroups[group], retVal);
							else
							{
								if (retVal != "") // use appending case
									retVal = String.format("%s %s", arabicAppendedGroup[group], retVal);
								else
									// use normal case
									retVal = String.format("%s %s", arabicGroup[group], retVal);
							}
						}
						else
							// use normal case
							retVal = String.format("%s %s", arabicGroup[group], retVal);

				}
				retVal = String.format("%s %s", groupDescription, retVal);
			}
			group++;
		}

		String formattedNumber = "";
		formattedNumber += String.format("%s ", ARABIC_PREFIX_TEXT);
		formattedNumber += (retVal != "") ? retVal : "";
		if (_intergerValue != 0)
		{
			// here we add currency name depending on _intergerValue : 1 ,2 ,
			// 3--->10 , 11--->99
			int remaining100 = (int) (_intergerValue % 100);
			if (remaining100 == 0)
				formattedNumber += currencyDetails.getArabic1CurrencyName();
			else if (remaining100 == 1)
				formattedNumber += currencyDetails.getArabic1CurrencyName();
			else if (remaining100 == 2)
			{
				if (_intergerValue == 2)
					formattedNumber += currencyDetails.getArabic2CurrencyName();
				else
					formattedNumber += currencyDetails.getArabic1CurrencyName();
			}
			else if (remaining100 >= 3 && remaining100 <= 10)
				formattedNumber += currencyDetails.getArabic310CurrencyName();
			else if (remaining100 >= 11 && remaining100 <= 99)
				formattedNumber += currencyDetails.getArabic1199CurrencyName();
		}

		formattedNumber += (_decimalValue != 0) ? " و " : "";
		formattedNumber += (_decimalValue != 0) ? decimalString : "";

		if (_decimalValue != 0)
		{
			// here we add currency part name depending on _intergerValue : 1 ,2
			// , 3--->10 , 11--->99
			formattedNumber += " ";
			int remaining100 = (int) (_decimalValue % 100);

			if (remaining100 == 0)
				formattedNumber += currencyDetails.getArabic1CurrencyPartName();
			else if (remaining100 == 1)
				formattedNumber += currencyDetails.getArabic1CurrencyPartName();
			else if (remaining100 == 2)
				formattedNumber += currencyDetails.getArabic2CurrencyPartName();
			else if (remaining100 >= 3 && remaining100 <= 10)
				formattedNumber += currencyDetails.getArabic310CurrencyPartName();
			else if (remaining100 >= 11 && remaining100 <= 99)
				formattedNumber += currencyDetails.getArabic1199CurrencyPartName();
		}
		formattedNumber += String.format(" %s", ARABIC_SUFFIX_TEXT);

		return formattedNumber;
	}

	private static String getDecimalValue(String decimalPart, CurrencyDetails currencyDetails)
	{
		String result = "";

		if (currencyDetails.getPartPrecision() != decimalPart.length())
		{
			int decimalPartLength = decimalPart.length();

			for (int i = 0; i < currencyDetails.getPartPrecision() - decimalPartLength; i++)
				// Fix for 1 number after decimal ( 10.5 ,1442.2 , 375.4 )
				decimalPart += "0";

			int dec = decimalPart.length() <= currencyDetails.getPartPrecision() ? decimalPart.length() : currencyDetails.getPartPrecision();
			result = decimalPart.substring(0, dec);
		}
		else
			result = decimalPart;

		for (int i = result.length(); i < currencyDetails.getPartPrecision(); i++)
			result += "0";

		return result;
	}

	private static String convertDecimalPartToEnWords(int groupNumber)
	{
		int tens = groupNumber % 100;
		int hundreds = groupNumber / 100;
		String retVal = "";

		if (hundreds > 0)
			retVal = String.format("%s %s", englishOnes[hundreds], englishGroup[0]);

		if (tens > 0)
			if (tens < 20)
				retVal += ((retVal != "") ? " " : "") + englishOnes[tens];
			else
			{
				int ones = tens % 10;
				tens = (tens / 10) - 2; // 20's offset
				retVal += ((retVal != "") ? " " : "") + englishTens[tens];

				if (ones > 0)
					retVal += ((retVal != "") ? " " : "") + englishOnes[ones];

			}

		return retVal;
	}

	private static String getDigitFeminineStatus(int digit, int groupLevel, CurrencyDetails currencyDetails)
	{
		if (groupLevel == -1)
		{ // if it is in the decimal part
			if (currencyDetails.isCurrencyPartNameFeminine())
				return arabicFeminineOnes[digit]; // use feminine field
			else
				return arabicOnes[digit];
		}
		else if (groupLevel == 0)
		{
			if (currencyDetails.isCurrencyNameFeminine())
				return arabicFeminineOnes[digit];// use feminine field
			else
				return arabicOnes[digit];
		}
		else
			return arabicOnes[digit];
	}

	private static String convertPartToArabicWords(int groupNumber, int groupLevel, BigDecimal remainingNumber, CurrencyDetails currencyDetails,
			long _intergerValue)
	{
		int tens = groupNumber % 100;
		int hundreds = groupNumber / 100;
		String retVal = "";

		if (hundreds > 0)
		{
			if (tens == 0 && hundreds == 2) // حالة المضاف
				retVal = String.format("%s", arabicAppendedTwos[0]);
			else
				// الحالة العادية
				retVal = String.format("%s", arabicHundreds[hundreds]);
		}

		if (tens > 0)
		{
			if (tens < 20)
			{
				// if we are processing under 20 numbers
				if (tens == 2 && hundreds == 0 && groupLevel > 0)
				{
					// This is special case for number 2 when it comes alone in
					// the group
					if (_intergerValue == 2000 || _intergerValue == 2000000 || _intergerValue == 2000000000 || _intergerValue == 2000000000000L
							|| _intergerValue == 2000000000000000L || _intergerValue == 2000000000000000000L)
						// في حالة الاضافة
						retVal = String.format("%s", arabicAppendedTwos[groupLevel]);
					else
						// في حالة الافراد
						retVal = String.format("%s", arabicTwos[groupLevel]);
				}
				else
				{
					// General case
					if (retVal != "")
						retVal += " و ";

					if (tens == 1 && groupLevel > 0 && hundreds == 0)
						retVal += " ";
					else if ((tens == 1 || tens == 2) && (groupLevel == 0 || groupLevel == -1) && hundreds == 0
							&& remainingNumber.compareTo(new BigDecimal(0)) == 0)
						retVal += ""; // Special case for 1 and 2 numbers like:
										// ليرة سورية و ليرتان سوريتان
					else
						retVal += getDigitFeminineStatus(tens, groupLevel, currencyDetails); // Get
					// Feminine status for this digit
				}
			}
			else
			{
				int ones = tens % 10;
				tens = (tens / 10) - 2; // 20's offset

				if (ones > 0)
				{
					if (retVal != "")
						retVal += " و ";

					// Get Feminine status for this digit
					retVal += getDigitFeminineStatus(ones, groupLevel, currencyDetails);
				}

				if (retVal != "")
					retVal += " و ";

				// Get Tens text
				retVal += arabicTens[tens];
			}
		}
		return retVal;
	}

}