package com.namasoft.common.utils.translation;

import com.namasoft.common.constants.Language;
import com.namasoft.common.utilities.*;

import java.text.MessageFormat;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Pattern;

public abstract class MessageTranslatorBase implements MessageTranslator
{
	protected AggregatedMessageTranslator aggregatedTranslator;

	@Override
	public String translate(Language lang, String message, Object... arguments)
	{
		String translatedMessage = translateMessage(lang, message);
		if (translatedMessage == null)
			return null;
		if (arguments == null || arguments.length == 0)
			return translatedMessage;
		return MessageFormat.format(translatedMessage, arguments);
	}

	public void processLang(Map<String, String> map, Language lang)
	{
		for (Entry<String, String> entry : map.entrySet())
		{
			processReference(entry, lang);
			processConcatenation(entry, lang);
		}
	}

	private void processReference(Entry<String, String> entry, Language lang)
	{
		if (entry.getValue().startsWith("@"))
		{
			String translatedValue = "";
			String[] parts = entry.getValue().split(Pattern.quote("@"));
			for (String part : parts)
			{
				if (ObjectChecker.isNotEmptyOrNull(part))
				{
					String translatedPart = translateFromParent(lang, part);
					if (translatedPart == null)
					{
						NaMaLogger.error("you did not define value for key {0}", part);
						translatedValue += part;
					}
					else
					{
						translatedValue += " " + translatedPart;
					}
				}
			}
			entry.setValue(translatedValue);
		}
	}

	private void processConcatenation(Entry<String, String> entry, Language lang)
	{
		if (entry == null)
			return;
		if (entry.getValue() == null)
		{
			NaMaLogger.error("you did not define value for key {0}", entry.getKey());
			return;
		}
		if (entry.getValue().startsWith("#"))
		{
			try
			{
				int part2Index = entry.getValue().indexOf("@");
				if (part2Index < 0)
					return;
				String part1 = entry.getValue().substring(1, part2Index);
				String part2 = entry.getValue().substring(part2Index + 1);
				entry.setValue(translateFromParent(lang, part1) + "|" + translateFromParent(lang, part2));
			}
			catch (Exception e)
			{
				e.printStackTrace();
			}
		}
	}

	private String translateFromParent(Language lang, String message)
	{
		if (aggregatedTranslator == null)
			return null;
		String mine = translate(lang, message);
		if (mine != null)
			return mine;
		return aggregatedTranslator.translateFromTranslators(lang, message, this);
	}

	public abstract String translateMessage(Language lang, String key);

}
