package com.namasoft.common.utils.translation;

import java.text.MessageFormat;
import java.util.*;
import java.util.stream.Collectors;

import com.namasoft.common.HasId;
import com.namasoft.common.constants.Language;
import com.namasoft.common.layout.TitledID;
import com.namasoft.common.layout.edit.NaMaText;
import com.namasoft.common.utilities.NaMaLogger;
import com.namasoft.common.utilities.ObjectChecker;
import com.namasoft.specialserialization.*;
import com.namasoft.common.utils.translation.TranslatorHelper.TranslationHelperResult;

public class TranslationUtil
{
	private static List<TranslatorHelper> helpers = null;

	public static MessageTranslator getTranslator()
	{
		return AggregatedMessageTranslator.instance();
	}

	public static String translateFieldId(Language lang, String fieldId)
	{
		if (ObjectChecker.isEmptyOrNull(fieldId))
			return "";
		if (!fieldId.contains("#"))
			return translate(lang, fieldId);
		return Arrays.stream(fieldId.split("#")).map(s -> translate(lang, s)).collect(Collectors.joining(" - "));
	}
	public static String translate(Language lang, String message, Object... arguments)
	{
		return getTranslator().translate(lang, message, arguments);
	}

	public static Object getName(Language lang, Object name1, Object name2)
	{
		if (lang == null)
			lang = Language.Arabic;
		switch (lang)
		{
		case Arabic:
			return ObjectChecker.getFirstNotNullObj(name1, name2);
		case English:
			return ObjectChecker.getFirstNotNullObj(name2, name1);
		}
		throw new RuntimeException(MessageFormat.format("Unhandled language constant: {0}", lang));
	}

	public static Object translateArgument(Language lang, Object object)
	{
		setupHelpers();
		for (TranslatorHelper helper : helpers)
		{
			TranslationHelperResult translation = helper.translate(lang, object);
			if (translation != null && translation.handled)
				return translation.translation;
		}
		return object;
	}

	private static synchronized void setupHelpers()
	{
		if (helpers == null)
		{
			helpers = new ArrayList<>();
			Set<Class<? extends TranslatorHelper>> helperClasses = ScanningUtils.getSubTypesOf(TranslatorHelper.class);
			ReflectionUtils.removeAbstractClasses(helperClasses);
			for (Class<? extends TranslatorHelper> klass : helperClasses)
			{
				try
				{
					helpers.add(ReflectionUtils.constructObject(klass));
				}
				catch (Exception e)
				{
					NaMaLogger.debug(e);
				}
			}
		}
	}

	public static Language getCurrentLanguage()
	{
		setupHelpers();
		for (TranslatorHelper helper : helpers)
		{
			Language currentLanguage = helper.getCurrentLanguage();
			if (currentLanguage != null)
				return currentLanguage;
		}
		return Language.Arabic;
	}

	public static String translate(NaMaText text)
	{
		return translate(text, getCurrentLanguage());
	}
	public static String translate(NaMaText text, Language lang)
	{
		if (ObjectChecker.isEmptyOrNull(text))
			return "";
		if (ObjectChecker.isNotEmptyOrNull(text.getResourceId()))
			return translate(lang, text.getResourceId());
		return lang == Language.Arabic ? text.getArabicText() : text.getEnglishText();
	}

	public static String translateTitledId(TitledID titledID, String prefix, Language lang)
	{
		if (ObjectChecker.isNotEmptyOrNull(titledID.getTitle()))
			return translate(titledID.getTitle(), lang);
		return translateId(titledID, prefix, lang);
	}

	public static String translateId(HasId hasId, String prefix, Language lang)
	{
		return translate(lang, prefix(prefix) + hasId.getId());
	}

	private static String prefix(String prefix)
	{
		if (ObjectChecker.isEmptyOrNull(prefix))
			return "";
		return prefix + ".";
	}

	public static void reset()
	{
		AggregatedMessageTranslator.instance().reset();
	}

}
