package com.namasoft.specialserialization;

import com.namasoft.common.*;
import com.namasoft.common.constants.DateWrapper;
import com.namasoft.common.flatobjects.*;
import com.namasoft.common.utilities.*;

import jakarta.activation.DataHandler;
import jakarta.xml.bind.annotation.XmlTransient;
import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;
import java.util.Map.Entry;

import static com.namasoft.specialserialization.ReflectionUtils.*;

@SuppressWarnings({ "rawtypes", "unchecked" })
public class FlatObjectUtilies
{
	public static <T, S> T convertBetweenCompatibleTypes(S source, Class<? extends T> klass)
	{
		if (source != null && klass.isAssignableFrom(source.getClass()))
			return (T) source;
		FlatObject flatSource = createFlatObject(source, 0);
		flatSource.setActualClassName(klass.getName());
		return (T) toRealObject(flatSource);
	}

	public static FlatObject createFlatObject(Object object)
	{
		return createFlatObject(object, 0);
	}

	public static FlatObject createFlatObject(Object object, long depth)
	{
		FlatObject flatObject = new FlatObject();
		if (object != null)
		{
			if (object instanceof FlatObject)
				return (FlatObject) object;
			flatObject.setActualClassName(object.getClass().getName());
			processObject(flatObject, object, depth + 1);
		}
		return flatObject;
	}

	private static void processObject(FlatObject flatObject, Object object, long depth)
	{
		if (isMap(object.getClass()))
		{
			processMapObject(flatObject, (Map) object, depth + 1);
		}
		else
		{
			processNormalObject(flatObject, object, depth + 1);
		}
	}

	private static void processMapObject(FlatObject flatObject, Map map, long depth)
	{
		for (Object entryObject : map.entrySet())
		{
			Entry entry = (Entry) entryObject;
			String id = entry.getKey().toString();
			Entry entryToAdd = getFlatObject(entry.getValue(), id, depth + 1);
			if (entryToAdd != null)
			{
				Object value = entryToAdd.getValue();
				if (value != null && value instanceof Date)
					value = new DateWrapper((Date) value);
				flatObject.getValuesMap().put(entryToAdd.getKey().toString(), value);
			}
		}
	}

	private static void addValuesEntry(FlatObject flatObject, Object object, Method method, long depth)
	{
		if (depth > 500)
			throw new RuntimeException("The depth exceeded 500 for object of type " + object.getClass() + " method " + method.getName());
		Object returnObject = callMethod(object, method);
		if (!isValidReturnObject(returnObject))
			return;
		String id = calculateId(method);
		Entry<String, Object> entry = getFlatObject(returnObject, id, depth + 1);
		if (entry != null)
		{
			Object value = entry.getValue();
			if (value != null && value instanceof Date)
				value = new DateWrapper((Date) value);
			flatObject.getValuesMap().put(entry.getKey(), value);
		}
	}

	private static boolean isValidReturnObject(Object returnObject)
	{
		if (returnObject == null)
			return false;
		if (DataHandler.class.isAssignableFrom(returnObject.getClass()))
			return false;
		return true;
	}

	private static Entry<String, Object> getFlatObject(Object returnObject, String id, long depth)
	{
		if (returnObject == null)
			return new SimpleEntry(id, null);
		Class<? extends Object> class1 = returnObject.getClass();
		if (CommonFlatObjectUtils.isNaMaPrimitive(class1) || class1.isAnonymousClass() || FlatObjectPrimitive.class.isAssignableFrom(class1))
		{
			if (returnObject instanceof Enum)
				return new SimpleEntry(id, ((Enum) returnObject).name());
			else
				return new SimpleEntry(id, returnObject);
		}
		if (isCollection(class1))
		{
			FlatObjectList flatList = processCollection((Collection) returnObject, id, depth + 1);
			return new SimpleEntry(id, flatList);
		}
		else
		{
			FlatObject flatObject = processComposite(returnObject, id, depth + 1);
			return new SimpleEntry(id, flatObject);
		}
	}

	private static FlatObject processComposite(Object returnObject, String id, long depth)
	{
		FlatObject flatObject = createFlatObject(returnObject, depth);
		return flatObject;
	}

	public static FlatObjectList processCollection(Collection returnList, String id)
	{
		return processCollection(returnList, id, 0);
	}

	public static FlatObjectList processCollection(Collection returnList, String id, long depth)
	{
		if (false && LoggingConfigurator.isInDebugMode() && ObjectChecker.isEmptyOrNull(returnList))
			NaMaLogger.error("An empty collection was sent to processCollection",
					new RuntimeException("An empty collection was sent to processCollection, might lead to IndexOutOfBoundsException in gwt"));
		FlatObjectList flatObjectList = new FlatObjectList();
		if (returnList == null)
			return flatObjectList;
		flatObjectList.setActualClassName(returnList.getClass().getName());
		for (Object object : returnList)
		{
			if (object instanceof HasSettableId && ObjectChecker.isEmptyOrNull(((HasSettableId) object).getId()))
				((HasSettableId) object).setId(SequentialUUID.guiUUID());
			flatObjectList.getList().add(getFlatObject(object, id, depth).getValue());
		}
		return flatObjectList;
	}

	private static String calculateId(Method method)
	{
		String fieldId = getFieldId(method);
		return fieldId;
	}

	private static String getFieldId(Method method)
	{
		String methodName = method.getName();
		String fieldName = "";
		if (methodName.startsWith("get"))
			fieldName = StringUtils.firstLetterLower(methodName.substring(3));
		else if (methodName.startsWith("is"))
			fieldName = StringUtils.firstLetterLower(methodName.substring(2));
		else
			throw new RuntimeException("Invalid method name " + methodName);
		return fieldName;
	}

	private static void processNormalObject(FlatObject flatObject, Object object, long depth)
	{
		ArrayList<Method> getterMethods = getGetterMethods(object.getClass());
		for (int i = 0; i < getterMethods.size(); i++)
		{
			addValuesEntry(flatObject, object, getterMethods.get(i), depth + 1);
		}
	}

	private static ArrayList<Method> getGetterMethods(Class<? extends Object> class1)
	{
		ArrayList<Method> getterMethods = getCahsedGetterMethods(class1.getName());
		if (getterMethods == null)
		{
			getterMethods = new ArrayList<Method>();
			Method[] methods = class1.getMethods();
			for (int i = 0; i < methods.length; i++)
			{
				if (isGetterMethod(methods[i]))
					getterMethods.add(methods[i]);
			}
			setCahsedGetterMethods(class1.getName(), getterMethods);
		}

		return getterMethods;
	}

	private static boolean shouldBeIgnored(Method method)
	{
		Annotation[] annotations = method.getDeclaredAnnotations();
		for (Annotation annotation : annotations)
		{
			if (Ignore.class.equals(annotation.annotationType()) || XmlTransient.class.equals(annotation.annotationType()))
			{
				return true;
			}
		}
		return false;
	}

	private static boolean isGetterMethod(Method method)
	{
		String methodName = method.getName();
		if (!methodName.startsWith("is") && !methodName.startsWith("get"))
		{
			return false;
		}
		if (methodName.equals("getClass"))
		{
			return false;
		}
		if (method.getParameterTypes().length > 0)
		{
			return false;
		}
		if (Void.class.equals(method.getReturnType()))
		{
			return false;
		}
		if (FlatObjectUtilies.shouldBeIgnored(method))
		{
			return false;
		}
		return true;
	}

	public static <T> T toRealObject(FlatObject flatObject)
	{
		return toRealObject(flatObject, true);
	}

	public static <T> T toRealObject(FlatObject flatObject, boolean removeEmptyObjectsFromLists)
	{
		try
		{
			if (flatObject == null || ObjectChecker.isEmptyOrNull(flatObject.getActualClassName()))
				return null;
			if (FlatObject.class.getName().equals(flatObject.getActualClassName()))
				return (T) flatObject;
			return toRealObject(flatObject, flatObject.getActualClassName(), removeEmptyObjectsFromLists);
		}
		catch (Exception e)
		{
			throw new RuntimeException("Techincal error happened", e);
		}
	}

	public static <T> T toRealObject(FlatObject flatObject, String className)
	{
		return toRealObject(flatObject, className, true);
	}

	public static <T> T toRealObject(FlatObject flatObject, String className, boolean removeEmptyObjectsFromLists)
	{
		if (flatObject == null)
			return null;
		try
		{
			if (FlatObject.class.getName().equals(className))
				return (T) flatObject;
			Object object = constructObject(className);
			for (Entry<String, Object> entry : flatObject.getValuesMap().entrySet())
			{
				setDirectValue(object, entry.getKey(), entry.getValue(), removeEmptyObjectsFromLists);
			}
			return (T) object;
		}
		catch (Exception e)
		{
			throw new RuntimeException("Techincal error happened", e);
		}
	}

	private static void setDirectValue(Object parent, String key, Object value, boolean removeEmptyObjectsFromLists) throws Exception
	{
		Object realValue = convertValueIfNeeded(parent, value, key, removeEmptyObjectsFromLists);
		if (isMap(parent.getClass()))
		{
			((Map) parent).put(key, realValue);
		}
		else
		{
			setProperty(parent, key, realValue);
		}
	}

	private static Object convertValueIfNeeded(Object parent, Object value, String key, boolean removeEmptyObjectsFromLists) throws Exception
	{
		Class suggestedSetterType = getSuggestedSetterType(parent, key);
		if (value instanceof FlatObjectList)
		{
			FlatObjectList flatObjectList = (FlatObjectList) value;
			if (removeEmptyObjectsFromLists)
				flatObjectList.removeEmptyLines();
			String actualClassName = flatObjectList.getActualClassName();
			if (parent == null || suggestedSetterType == null)
			{
				actualClassName = ArrayList.class.getName();
			}
			else
			{
				actualClassName = suggestedSetterType.getName();
			}
			return toRealObject(flatObjectList, actualClassName);
		}
		if (value instanceof FlatObject && !FlatObject.class.equals(suggestedSetterType))
		{
			FlatObject flatObject = (FlatObject) value;
			String actualClassName = flatObject.getActualClassName();
			if (actualClassNameIsEmpty(actualClassName))
			{
				if (parent == null || suggestedSetterType == null)
				{
					return value;
				}
				else
				{
					actualClassName = suggestedSetterType.getName();
				}
			}
			return toRealObject(flatObject, actualClassName);
		}
		if (suggestedSetterType != null && Enum.class.isAssignableFrom(suggestedSetterType))
		{
			return suggestedSetterType.getMethod("valueOf", String.class).invoke(null, value);
		}
		if (value != null && value instanceof DateWrapper)
			value = ((DateWrapper) value).toDate();
		return value;
	}

	protected static boolean actualClassNameIsEmpty(String actualClassName)
	{
		return actualClassName == null || actualClassName.isEmpty();
	}

	public static List toRealList(FlatObjectList flatObjectList)
	{
		return (List) toRealObject(flatObjectList, ArrayList.class.getName());
	}

	public static Object toRealObject(FlatObjectList flatObjectList, String className)
	{
		return toRealObject(flatObjectList, className, true);
	}

	public static Object toRealObject(FlatObjectList flatObjectList, String className, boolean removeEmptyObjectsFromLists)
	{
		if (flatObjectList == null)
			return null;
		try
		{
			Collection collection = (Collection) ReflectionUtils.constructObject(className);
			for (int i = 0; i < flatObjectList.getList().size(); i++)
			{
				collection.add(convertValueIfNeeded(null, flatObjectList.getList().get(i), "", removeEmptyObjectsFromLists));
			}
			return collection;
		}
		catch (Exception e)
		{
			throw new RuntimeException(e);
		}

	}

	public static FlatObjectList emptyList(String fieldId)
	{
		return FlatObjectUtilies.processCollection(new ArrayList<>(), fieldId, 0);
	}

	public static <T> T cloneObject(T original)
	{
		return (T) toRealObject(createFlatObject(original, 0));
	}

    public static <T> List<T> convertListToAnotherType(List lines, Class<T> klass)
    {
		if (ObjectChecker.isEmptyOrNull(lines))
			return new ArrayList<>();
		List<T> ret = new ArrayList<>(lines.size());
		for (Object o : lines)
		{
			ret.add(convertBetweenCompatibleTypes(o, klass));
		}
		return ret;
	}

	public static void updateDetailsClassName(FlatObject flatObject, String fieldName, Class<?> klass)
	{
		if (flatObject == null || !(flatObject.getFieldValue(fieldName) instanceof FlatObjectList))
			return;
		FlatObjectList list = flatObject.getFieldValue(fieldName);
		if (list.getList() == null)
			return;
		for (Object line : list.getList())
		{
			if (!(line instanceof FlatObject))
				continue;
			((FlatObject) line).setActualClassName(klass.getName());
		}
	}

	private static class SimpleEntry<K, V> implements Entry<K, V>, Serializable
	{
		private static final long serialVersionUID = 4052390067508552701L;
		private final K key;
		private V value;

		public SimpleEntry(K key, V value)
		{
			this.key = key;
			this.value = value;
		}

		@Override
		public K getKey()
		{
			return key;
		}

		@Override
		public V getValue()
		{
			return value;
		}

		@Override
		public V setValue(V value)
		{
			return null;
		}
	}

	public static List<FlatObject> convertToListOfFlatObjects(List original)
	{
		List<FlatObject> flat = new ArrayList<>();
		if (ObjectChecker.isNotEmptyOrNull(original))
			for (Object object : original)
				flat.add(createFlatObject(object, 0));
		return flat;
	}

	public static List listOfFLatToReal(List<FlatObject> original, String className)
	{
		List real = new ArrayList<>();
		if (ObjectChecker.isNotEmptyOrNull(original))
			for (FlatObject object : original)
			{
				if (ObjectChecker.isNotEmptyOrNull(className))
					real.add(toRealObject(object, className));
				else
					real.add(toRealObject(object));
			}
		return real;
	}

	public static <T extends Enum<T>> T tryParse(Class<T> enumType, String name)
	{
		if (ObjectChecker.isEmptyOrNull(name))
			return null;
		name = name.trim();
		try
		{
			return Enum.valueOf(enumType, name);
		}
		catch (Exception e)
		{
			return null;
		}
	}

}
