package com.namasoft.specialserialization;

import com.namasoft.common.Pair;
import com.namasoft.common.utilities.*;
import io.github.classgraph.*;
import io.github.toolfactory.narcissus.Narcissus;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

public class ScanningUtils
{
	private final static Map<String, Pair<ScanResult, LocalDateTime>> reflectionsMap = new ConcurrentHashMap<>();
	private static ScheduledExecutorService reflectionsMapCleaner;
	private static boolean useReflectionsMapCleaner = true;

	public static synchronized void doNotUseReflectionsMapCleaner()
	{
		useReflectionsMapCleaner = false;
	}

	public static boolean isUseReflectionsMapCleaner()
	{
		return useReflectionsMapCleaner;
	}

	public static synchronized ScanResult createReflectionScanner()
	{
		return createReflectionScanner("com.namasoft");
	}

	public static synchronized ScanResult createReflectionScanner(String packageName)
	{
		Pair<ScanResult, LocalDateTime> scanResult = reflectionsMap.get(packageName);
		if (scanResult != null)
			return getResult(scanResult);
		NaMaProfiler.get().start("creating reflections for " + packageName);
		ScanResult result = new ClassGraph().verbose(false).enableAllInfo().acceptPackages(packageName).scan();
		reflectionsMap.put(packageName, scanResult = new Pair<>(result, LocalDateTime.now()));
		NaMaProfiler.get().end("creating reflections for " + packageName);
		return getResult(scanResult);
	}

	private static ScanResult getResult(Pair<ScanResult, LocalDateTime> result)
	{
		if (useReflectionsMapCleaner && reflectionsMapCleaner == null)
		{
			reflectionsMapCleaner = Executors.newSingleThreadScheduledExecutor(Thread.ofVirtual().name("reflectionsMapCleaner").factory());
			reflectionsMapCleaner.scheduleWithFixedDelay(ScanningUtils::removeExpiredResults, 10, 10, TimeUnit.MINUTES);
		}
		result.setY(LocalDateTime.now());
		return result.getX();
	}

	private static void removeExpiredResults()
	{
		Set<Map.Entry<String, Pair<ScanResult, LocalDateTime>>> toRemove = reflectionsMap.entrySet().stream()
				.filter(e -> e.getValue().getY().until(LocalDateTime.now(), ChronoUnit.MINUTES) > 15).collect(Collectors.toSet());
		reflectionsMap.entrySet().removeAll(toRemove);
		toRemove.forEach(e -> e.getValue().getX().close());
		if (reflectionsMap.isEmpty())
		{
			reflectionsMapCleaner.close();
			reflectionsMapCleaner = null;
		}
	}

	public static void closeScanResults()
	{
		reflectionsMap.values().forEach(e -> e.getX().close());
		reflectionsMap.clear();
	}

	public static Set<Method> getMethodsAnnotatedWith(Class<? extends Annotation> annotation)
	{
		ScanResult scanResult = createReflectionScanner();
		List<Class<?>> classes = scanResult.getClassesWithMethodAnnotation(annotation).loadClasses();
		Set<Method> methods = new HashSet<>();
		for (Class<?> klass : classes)
		{
			Set<Method> methodSet = Arrays.stream(Narcissus.getDeclaredMethods(klass)).collect(Collectors.toSet());
			methodSet.removeIf(m -> Arrays.stream(m.getDeclaredAnnotations())
					.noneMatch(a -> ObjectChecker.areEqual(a.annotationType().getName(), annotation.getName())));
			methods.addAll(methodSet);
		}
		return methods;
	}

	public static <T> Set<Class<? extends T>> getSubTypesOf(Class<T> type)
	{
		ScanResult scanResult = createReflectionScanner();
		return getSubTypesOf(scanResult, type);
	}

	public static <T> Set<Class<? extends T>> getSubTypesOf(ScanResult scanResult, Class<T> type)
	{
		if (type.isInterface())
			return CollectionsUtility.castSet(new HashSet<>(scanResult.getClassesImplementing(type).loadClasses()));
		return CollectionsUtility.castSet(new HashSet<>(scanResult.getSubclasses(type).loadClasses()));
	}

	public static Set<Class<?>> getTypesAnnotatedWith(Class<? extends Annotation> annotation)
	{
		return getTypesAnnotatedWith(createReflectionScanner(), annotation);
	}

	public static Set<Class<?>> getTypesAnnotatedWith(ScanResult scanResult, Class<? extends Annotation> annotation)
	{
		return CollectionsUtility.castSet(new HashSet<>(scanResult.getClassesWithAnnotation(annotation).loadClasses()));
	}

	public static List<FieldInfo> getFieldsAnnotatedWith(ScanResult scanResult, Class<? extends Annotation> annotation)
	{
		ClassInfoList cs = scanResult.getClassesWithFieldAnnotation(annotation);
		List<FieldInfo> fields = new ArrayList<>();
		for (ClassInfo c : cs)
		{
			for (FieldInfo info : c.getDeclaredFieldInfo())
			{
				if (info.getAnnotationInfo(annotation) != null)
					fields.add(info);
			}
		}
		return fields;
	}
}
