package com.namasoft.specialserialization;

import com.namasoft.common.utilities.ObjectChecker;
import com.namasoft.common.utils.PlatformInfo;

import javax.imageio.ImageIO;
import javax.swing.*;
import javax.swing.border.EmptyBorder;
import java.awt.*;
import java.awt.event.*;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Timer;
import java.util.TimerTask;

public class ScreenLocker extends JFrame
{
	private JPanel contentPane;
	private JLabel lblStatus;
	private TrayIcon trayIcon;
	private final ScreenLockerCallback callback;

	public ScreenLocker(ScreenLockerCallback callback) throws HeadlessException
	{
		this.callback = callback;
		setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
		setBounds(100, 100, 608, 150);
		contentPane = new JPanel();
		contentPane.setBorder(new EmptyBorder(5, 5, 5, 5));
		setContentPane(contentPane);
		contentPane.setLayout(null);

		lblStatus = new JLabel("Status");
		lblStatus.setBounds(12, 12, 500, 20);
		contentPane.add(lblStatus);
		JButton hide = new JButton("Hide");
		hide.setBounds(12, 40, 100, 40);
		contentPane.add(hide);
		hide.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e)
			{
				setVisible(false);
			}
		});
		JButton exit = new JButton("Exit");
		exit.setBounds(115, 40, 100, 40);
		if (callback.shouldAddExitButton())
			contentPane.add(exit);

		exit.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e)
			{
				System.exit(0);
			}
		});
		if (callback.shouldAddSystemTray())
			createTray();
		Timer timer = new Timer();
		timer.scheduleAtFixedRate(new TimerTask()
		{
			@Override
			public void run()
			{
				try
				{
					lockScreen();
				}
				catch (Exception ignored)
				{
					ignored.printStackTrace();
				}
			}
		}, callback.initialDelay(), callback.period());
		updateSatus(true);
	}

	private void createTray()
	{
		MediaTracker tracker = new MediaTracker(this);
		Image image = null;
		try
		{
			InputStream resourceAsStream = getClass().getClassLoader().getResourceAsStream("locker.jpg");
			image = ImageIO.read(resourceAsStream);
			resourceAsStream.close();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		tracker.addImage(image, 0);
		try
		{
			tracker.waitForAll();
		}
		catch (InterruptedException e1)
		{
			e1.printStackTrace();
		}
		final PopupMenu popup = new PopupMenu();
		trayIcon = new TrayIcon(image, "Nama Screen Locker daemon");
		final SystemTray tray = SystemTray.getSystemTray();

		// Create a pop-up menu components
		final CheckboxMenuItem showItem = new CheckboxMenuItem("Display");
		MenuItem exitItem = new MenuItem("Exit");
		exitItem.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e)
			{
				System.exit(0);
			}
		});
		showItem.addItemListener(new ItemListener()
		{
			@Override
			public void itemStateChanged(ItemEvent e)
			{
				setVisible(showItem.getState());
			}
		});
		// Add components to pop-up menu
		popup.add(showItem);
		popup.add(exitItem);

		trayIcon.setPopupMenu(popup);
		trayIcon.addActionListener(new ActionListener()
		{
			@Override
			public void actionPerformed(ActionEvent e)
			{
				setVisible(!isVisible());
			}
		});
		try
		{
			tray.add(trayIcon);
		}
		catch (AWTException e)
		{
			System.out.println("TrayIcon could not be added.");
		}
		addWindowListener(new WindowAdapter()
		{
			@Override
			public void windowClosing(WindowEvent e)
			{
				showItem.setState(false);
			}
		});
	}

	public static void main(String[] args) throws IOException
	{
		ScreenLocker locker = new ScreenLocker(ScreenLockerCallback.NULL);
		locker.setVisible(true);
	}

	private void lockScreen() throws IOException
	{
		if (!shouldLock())
			return;
		updateSatus(false);
		if (PlatformInfo.isLinux())
			new ProcessBuilder("gnome-screensaver-command", "-l").start();
		else
			new ProcessBuilder("rundll32.exe", "user32.dll,LockWorkStation").start();
	}

	private boolean shouldLock()
	{
		return callback.shouldLock();
	}

	private void updateSatus(boolean firstTime)
	{
		Calendar time = Calendar.getInstance();
		SimpleDateFormat format = new SimpleDateFormat("HH:mm");
		String now = format.format(time.getTime());
		time.add(Calendar.MINUTE, 20);
		String next = format.format(time.getTime());
		String text = "Last lock on " + now + ", next lock on " + next;
		lblStatus.setText(ObjectChecker.getFirstNotEmptyObj(callback.statusText(), text));
		if (trayIcon != null)
		{
			trayIcon.setToolTip("Nama Screen Locker: " + text);
			setTitle(trayIcon.getToolTip());
		}
		if (firstTime)
		{
			if (callback.shouldStartVisible())
				setVisible(true);
		}
		else if (callback.shouldShowOnLock())
		{
			setVisible(true);
		}
	}

	public static long minutes(long minutes)
	{
		return minutes * 60 * 1000;
	}

	public interface ScreenLockerCallback
	{
		ScreenLockerCallback NULL = new ScreenLockerCallback()
		{
			@Override
			public boolean shouldLock()
			{
				return true;
			}

			@Override
			public boolean shouldAddSystemTray()
			{
				return true;
			}

			@Override
			public boolean shouldShowOnLock()
			{
				return true;
			}

			@Override
			public long initialDelay()
			{
				return minutes(10);
			}

			@Override
			public long period()
			{
				return minutes(20);
			}

			@Override
			public String statusText()
			{
				return null;
			}

			@Override
			public boolean shouldStartVisible()
			{
				return true;
			}

			@Override
			public boolean shouldAddExitButton()
			{
				return true;
			}
		};

		boolean shouldLock();

		boolean shouldAddSystemTray();

		boolean shouldShowOnLock();

		long initialDelay();

		long period();

		String statusText();

		boolean shouldStartVisible();

		boolean shouldAddExitButton();
	}
}

