package com.namasoft.taxauthority.ereceipt;

import com.namasoft.common.utilities.*;
import com.namasoft.taxauthority.*;
import jakarta.xml.bind.annotation.*;

import java.math.BigDecimal;
import java.util.*;

@XmlRootElement
@XmlAccessorType(XmlAccessType.PROPERTY)
public class EReceiptLine extends EInvoiceDocLine
{
	private String internalCode;
	private String description;
	private String itemType;
	private String itemCode;
	private String unitType;
	private BigDecimal quantity;
	private BigDecimal unitPrice;
	private BigDecimal netSale;
	private BigDecimal totalSale;
	private BigDecimal total;
	private BigDecimal valueDifference = BigDecimal.ZERO;
	private List<EReceiptDiscount> commercialDiscountData;
	private List<EReceiptDiscount> itemDiscountData;
	private List<TaxAuthorityLineTax> taxableItems;

	public String getInternalCode()
	{
		return internalCode;
	}

	public void setInternalCode(String internalCode)
	{
		this.internalCode = internalCode;
	}

	public String getDescription()
	{
		return description;
	}

	public void setDescription(String description)
	{
		this.description = description;
	}

	public String getItemType()
	{
		return itemType;
	}

	public void setItemType(String itemType)
	{
		this.itemType = itemType;
	}

	public String getItemCode()
	{
		return itemCode;
	}

	public void setItemCode(String itemCode)
	{
		this.itemCode = itemCode;
	}

	public String getUnitType()
	{
		return unitType;
	}

	public void setUnitType(String unitType)
	{
		this.unitType = unitType;
	}

	public BigDecimal getQuantity()
	{
		return quantity;
	}

	@Override
	public void setQuantity(BigDecimal quantity)
	{
		this.quantity = quantity;
	}

	public BigDecimal getUnitPrice()
	{
		return unitPrice;
	}

	public void setUnitPrice(BigDecimal unitPrice)
	{
		this.unitPrice = unitPrice;
	}

	public BigDecimal getNetSale()
	{
		return netSale;
	}

	public void setNetSale(BigDecimal netSale)
	{
		this.netSale = netSale;
	}

	public BigDecimal getTotalSale()
	{
		return totalSale;
	}

	public void setTotalSale(BigDecimal totalSale)
	{
		this.totalSale = totalSale;
	}

	public BigDecimal getTotal()
	{
		return total;
	}

	public void setTotal(BigDecimal total)
	{
		this.total = total;
	}

	public BigDecimal getValueDifference()
	{
		return valueDifference;
	}

	public void setValueDifference(BigDecimal valueDifference)
	{
		this.valueDifference = valueDifference;
	}

	public List<EReceiptDiscount> getCommercialDiscountData()
	{
		return commercialDiscountData;
	}

	public void setCommercialDiscountData(List<EReceiptDiscount> commercialDiscountData)
	{
		this.commercialDiscountData = commercialDiscountData;
	}

	public List<EReceiptDiscount> getItemDiscountData()
	{
		return itemDiscountData;
	}

	public void setItemDiscountData(List<EReceiptDiscount> itemDiscountData)
	{
		this.itemDiscountData = itemDiscountData;
	}

	public List<TaxAuthorityLineTax> getTaxableItems()
	{
		return taxableItems;
	}

	public void setTaxableItems(List<TaxAuthorityLineTax> taxableItems)
	{
		this.taxableItems = taxableItems;
	}

	@Override
	public void calcSalesTotal()
	{
		setTotalSale(NaMaMath.multiply(getQuantity(), getUnitPrice()));
	}

	@Override
	public void updateUnitPrice(TaxAuthorityUnitValue unitValue)
	{
		setUnitPrice(ObjectChecker.getFirstNotEmptyOrZero(unitValue.getAmountSold(), unitValue.getAmountEGP()));
	}

	@Override
	public void updateSalesTotal(BigDecimal salesTotal)
	{
		setTotalSale(salesTotal);
	}

	@Override
	public BigDecimal fetchSalesTotal()
	{
		return getTotalSale();
	}

	@Override
	public void calcNetTotal()
	{
		setNetSale(NaMaMath.zeroIfNull(getTotalSale()).subtract(NaMaMath.zeroIfNull(fetchDiscount().getAmount())));
	}

	@Override
	public void updateNetTotal(BigDecimal netTotal)
	{
		setNetSale(netTotal);
	}

	@Override
	public BigDecimal fetchNetTotal()
	{
		return getNetSale();
	}

	@Override
	public void calcTotal()
	{
		setTotal(totalWithoutDiscount());
	}

	private BigDecimal totalWithoutDiscount()
	{
		BigDecimal totalTaxAmount = NaMaMath.totalizeDecimalStream(getTaxableItems().stream()
				.map(l -> ObjectChecker.isTrue(l.getDiscount()) ? NaMaMath.multiply(l.getAmount(), BigDecimal.valueOf(-1)) : l.getAmount()));
		return NaMaMath.sum(netSale, totalTaxAmount, valueDifference);
	}

	@Override
	public void updateTotal(BigDecimal total)
	{
		setTotal(total);
	}

	@Override
	public BigDecimal fetchFinalTotal()
	{
		return getTotal();
	}

	@Override
	public void updateDiscount(TaxAuthorityDiscount discount)
	{
		EReceiptDiscount first = CollectionsUtility.getFirst(getCommercialDiscountData());
		if (first == null)
		{
			setCommercialDiscountData(new ArrayList<>());
			first = new EReceiptDiscount();
			getCommercialDiscountData().add(first);
		}
		first.setAmount(discount.getAmount());
		first.setDescription("Commercial Discount");
	}

	@Override
	public TaxAuthorityDiscount fetchDiscount()
	{
		EReceiptDiscount first = CollectionsUtility.getFirst(getCommercialDiscountData());
		TaxAuthorityDiscount discount = new TaxAuthorityDiscount();
		if (first == null)
			return discount;
		discount.setAmount(first.getAmount());
		discount.setRate(NaMaMath.calcPercentage(getTotalSale(), first.getAmount()));
		return discount;
	}

	@Override
	public void updateTaxableItems(List<TaxAuthorityLineTax> taxableItems)
	{
		setTaxableItems(taxableItems);
	}

	@Override
	public List<TaxAuthorityLineTax> fetchTaxableItems()
	{
		return getTaxableItems();
	}

	@Override
	public void updateTotalTaxableFees(BigDecimal totalTaxableFees)
	{

	}

	@Override
	public void updateItemsDiscount(BigDecimal totalTaxableFees)
	{

	}

	@Override
	public void updateItemCode(String itemCode)
	{
		setItemCode(itemCode);
	}

	@Override
	public String fetchItemCode()
	{
		return getItemCode();
	}

	@Override
	public void updateItemType(String itemType)
	{
		setItemType(itemType);
	}

	@Override
	public void updateInternalCode(String internalCode)
	{
		setInternalCode(internalCode);
	}

	@Override
	public void updateDescription(String description)
	{
		setDescription(description);
	}

	@Override
	public BigDecimal fetchItemsDiscount()
	{
		return BigDecimal.ZERO;
	}

	@Override
	public void updateUnitType(String unitType)
	{
		setUnitType(unitType);
	}

	@Override
	public String fetchUnitType()
	{
		return getUnitType();
	}

	@Override
	public String currencySold(EInvoiceDoc doc)
	{
		return doc.fetchHeaderCurrency();
	}

	@Override
	public BigDecimal currencyRate(EInvoiceDoc doc)
	{
		return doc.fetchHeaderCurrencyRate();
	}

	@Override
	public String fetchItemType()
	{
		return getItemType();
	}

	@Override
	public String fetchDescription()
	{
		return getDescription();
	}

	@Override
	public String fetchInternalCode()
	{
		return getInternalCode();
	}

	@Override
	public BigDecimal fetchUnitPrice()
	{
		return getUnitPrice();
	}
}
