package com.namasoft.contracts.common.dtos;
import com.namasoft.common.fieldids.newids.basic.IdsOfEntityTypeList;

import com.namasoft.common.flatobjects.EntityReferenceData;
import com.namasoft.common.utilities.ObjectChecker;
import com.namasoft.common.utilities.TrueFalseNotSpecified;
import com.namasoft.modules.commonbasic.contracts.valueobjects.IDTOHasEntities;
import com.namasoft.modules.commonbasic.contracts.entities.DTOSecurityProfile;
import com.namasoft.modules.commonbasic.contracts.valueobjects.*;
import com.namasoft.modules.commonbasic.enums.ApplicableWhen;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlRootElement;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

@XmlRootElement
@XmlAccessorType(XmlAccessType.PROPERTY)
public class DTOUser extends MasterFileDTO
{
	// TODO: userCanAction must be implemented and support for actions
	private static final long serialVersionUID = -5853811953783590084L;
	private String loginId;
	private String password;
	private String email;
	private Integer checkNotificationPeriod;
	private EntityReferenceData securityProfile;
	private EntityReferenceData defaultLegalEntity;
	private EntityReferenceData defaultMenu;
	private EntityReferenceData defaultShortCuts;
	private DTOSecurityProfile fullProfile;
	private EntityReferenceData employee;
	private DTOUserSettings settings;
	private List<DTOUser> alternativeUsers;

	private String preferredLanguage;

	public String getPreferredLanguage()
	{
		return preferredLanguage;
	}

	public void setPreferredLanguage(String preferredLanguage)
	{
		this.preferredLanguage = preferredLanguage;
	}

	public DTOUser()
	{
		super();
	}

	public String getLoginId()
	{
		return loginId;
	}

	public void setLoginId(String loginId)
	{
		this.loginId = loginId;
	}

	public String getPassword()
	{
		return password;
	}

	public void setPassword(String password)
	{
		this.password = password;
	}

	public String getEmail()
	{
		return email;
	}

	public void setEmail(String email)
	{
		this.email = email;
	}

	public EntityReferenceData getSecurityProfile()
	{
		return securityProfile;
	}

	public void setSecurityProfile(EntityReferenceData securityProfile)
	{
		this.securityProfile = securityProfile;
	}

	public DTOSecurityProfile getFullProfile()
	{
		return fullProfile;
	}

	public void setFullProfile(DTOSecurityProfile fullProfile)
	{
		this.fullProfile = fullProfile;
	}

	public boolean userCan(String entityType, String capabilityId)
	{
		if (getFullProfile() == null || getFullProfile().getStandardLines() == null)
			return false;
		if (entityType == null || capabilityId == null)
			return false;
		if (getFullProfile().getFullAuthority())
			return true;
		List<DTOStandardSecurityLine> standardLines = getFullProfile().getStandardLines();
		Boolean userCan = userCan(entityType, capabilityId, getSettings().getStandardLines());
		if (userCan != null)
			return userCan;
		userCan = userCan(entityType, capabilityId, standardLines);
		if (userCan != null)
			return userCan;

		return false;
	}

	public Integer minEntitySearchQueryLength(String entityType)
	{
		if (getFullProfile() == null || getFullProfile().getStandardLines() == null || getFullProfile().getFullAuthority())
			return -1;
		if (entityType == null)
			return -1;
		Integer minSearchQueryLength = findMinSearchQueryLength(entityType, getSettings().getStandardLines());
		if (minSearchQueryLength > 0)
			return minSearchQueryLength;
		return findMinSearchQueryLength(entityType, getFullProfile().getStandardLines());
	}

	private Integer findMinSearchQueryLength(String entityType, List<? extends DTOAbstractStandardSecurityLine> standardLines)
	{
		if (ObjectChecker.isEmptyOrNull(standardLines))
			return -1;
		List<DTOAbstractStandardSecurityLine> linesWithMinSearchQueryLength = standardLines.stream()
				.filter(l -> ObjectChecker.isNotEmptyOrZero(l.getMinSearchQueryLength())).collect(Collectors.toList());
		if (ObjectChecker.isEmptyOrNull(linesWithMinSearchQueryLength))
			return -1;
		Optional<DTOAbstractStandardSecurityLine> firstLineMatched = linesWithMinSearchQueryLength.stream()
				.filter(l -> ObjectChecker.areEqual(l.fetchTargetEntity(), entityType)).findFirst();
		if (firstLineMatched.isPresent())
			return firstLineMatched.get().getMinSearchQueryLength();
		firstLineMatched = linesWithMinSearchQueryLength.stream().filter(l -> entityTypeListContainsEntity(l.fetchTargetEntities(), entityType))
				.findFirst();
		if (firstLineMatched.isPresent())
			return firstLineMatched.get().getMinSearchQueryLength();
		firstLineMatched = linesWithMinSearchQueryLength.stream()
				.filter(l -> ObjectChecker.areAllEmptyOrNull(l.fetchTargetEntities(), l.fetchTargetEntity())).findFirst();
		if (firstLineMatched.isPresent())
			return firstLineMatched.get().getMinSearchQueryLength();
		return -1;

	}
	private <T extends DTOAbstractStandardSecurityLine & IDTOHasEntities> Boolean userCan(String entityType, String capabilityId,
			List<T> standardLines)
	{
		for (T userCapability : standardLines)
		{
			if (ObjectChecker.areEqual(userCapability.getTargetEntity(), entityType))
				return userCapability.userCan(capabilityId);
		}
		for (T userCapability : standardLines)
		{
			if (entityTypeListContainsEntity(userCapability.getTargetEntities(), entityType))
				return userCapability.userCan(capabilityId);
		}
		for (T userCapability : standardLines)
		{
			if (ObjectChecker.areAllEmptyOrNull(userCapability.getTargetEntities(), userCapability.getTargetEntity()))
				return userCapability.userCan(capabilityId);
		}
		return null;
	}

	private boolean entityTypeListContainsEntity(EntityReferenceData targetEntities, String entityType)
	{
		return ObjectChecker.isNotEmptyOrNull(targetEntities)
				&& ObjectChecker.isNotEmptyOrNull(targetEntities.get(IdsOfEntityTypeList.lines_entityType))
				&& targetEntities.<String> get(IdsOfEntityTypeList.lines_entityType).contains("," + entityType + ",");
	}

	public Integer getCheckNotificationPeriod()
	{
		return checkNotificationPeriod;
	}

	public void setCheckNotificationPeriod(Integer checkNotificationPeriod)
	{
		this.checkNotificationPeriod = checkNotificationPeriod;
	}

	public EntityReferenceData getDefaultLegalEntity()
	{
		return defaultLegalEntity;
	}

	public void setDefaultLegalEntity(EntityReferenceData defaultLegalEntity)
	{
		this.defaultLegalEntity = defaultLegalEntity;
	}

	public EntityReferenceData getDefaultMenu()
	{
		return defaultMenu;
	}

	public void setDefaultMenu(EntityReferenceData defaultMenu)
	{
		this.defaultMenu = defaultMenu;
	}

	public boolean userCanViewField(String targetEntity, String fieldId)
	{
		if (getFullProfile() == null)
			return false;
		if (ObjectChecker.areAllEmptyOrNull(getSettings().getDisabledFields(), getFullProfile().getDisabledFields()))
			return true;
		if (targetEntity == null || fieldId == null)
			return true;
		if (ObjectChecker.isTrue(getFullProfile().getFullAuthority()))
			return true;

		DTOAbstractSecurityFieldAuthority authority = getFieldSecurityLine(targetEntity, fieldId, ApplicableWhen.All);
		if (authority == null)
			return true;
		return ObjectChecker.areNotEqual(authority.getAuthorityType(), DTOSecurityFieldAuthority.InVisible);
	}

	private DTOAbstractSecurityFieldAuthority getFieldSecurityLine(String targetEntity, String fieldId, ApplicableWhen applicableWhen)
	{
		DTOAbstractSecurityFieldAuthority line = calcFieldLineFromLines(targetEntity, fieldId, applicableWhen, getSettings().getDisabledFields());
		if (line != null)
			return line;
		return calcFieldLineFromLines(targetEntity, fieldId, applicableWhen, getFullProfile().getDisabledFields());
	}

	private DTOAbstractSecurityFieldAuthority calcFieldLineFromLines(String targetEntity, String fieldId, ApplicableWhen applicableWhen,
			List<DTOSecurityFieldAuthority> disabledFields)
	{
		if (ObjectChecker.isEmptyOrNull(disabledFields))
			return null;
		DTOAbstractSecurityFieldAuthority wildCardLine = null;
		for (DTOSecurityFieldAuthority authority : disabledFields)
		{
			if (modeApplicable(authority, applicableWhen) && sameEntity(targetEntity, authority.getEntityType()))
			{
				if (ObjectChecker.areEqual(fieldId, authority.getFieldId()))
					return authority;
				if (wildCardLine == null && ObjectChecker.areEqual("*", authority.getFieldId()))
					wildCardLine = authority;
			}

		}
		for (DTOSecurityFieldAuthority authority : disabledFields)
		{
			if (modeApplicable(authority, applicableWhen) && entityTypeListContainsEntity(authority.getTargetEntities(), targetEntity))
			{
				if (ObjectChecker.areEqual(fieldId, authority.getFieldId()))
					return authority;
				if (wildCardLine == null && ObjectChecker.areEqual("*", authority.getFieldId()))
					wildCardLine = authority;
			}
		}
		if (wildCardLine == null)
		{
			for (DTOSecurityFieldAuthority authority : disabledFields)
			{
				if (modeApplicable(authority, applicableWhen) && ObjectChecker
						.areAllEmptyOrNull(authority.getTargetEntities(), authority.getEntityType()))
				{
					if (ObjectChecker.areEqual(fieldId, authority.getFieldId()))
						return authority;
					if (wildCardLine == null && ObjectChecker.areEqual("*", authority.getFieldId()))
						wildCardLine = authority;
				}
			}
		}
		return wildCardLine;
	}

	private boolean modeApplicable(DTOSecurityFieldAuthority authority, ApplicableWhen applicableWhen)
	{
		switch (applicableWhen)
		{
		case NewMode:
			return ObjectChecker.isTrue(authority.getApplicableWithNewMode());
		case UpdateMode:
			return ObjectChecker.isTrue(authority.getApplicableWithUpdateMode());
		case DraftMode:
			return ObjectChecker.isTrue(authority.getApplicableWithDraftMode());
		}
		return ObjectChecker
				.areAllTrue(authority.getApplicableWithNewMode(), authority.getApplicableWithUpdateMode(), authority.getApplicableWithDraftMode());
	}

	public String getPageAuthority(String targetEntity, String pageId, int pageNumber)
	{
		if (getFullProfile() == null)
			return DTOSecurityFieldAuthority.Normal;
		if (targetEntity == null)
			return DTOSecurityFieldAuthority.Normal;
		if (ObjectChecker.isTrue(getFullProfile().getFullAuthority()))
			return DTOSecurityFieldAuthority.Normal;
		if (ObjectChecker.areAllEmptyOrNull(getSettings().getPageSecurity(), getFullProfile().getPageSecurity()))
			return DTOSecurityFieldAuthority.NotDefined;
		String pageSecurity = calcPageSecurity(targetEntity, pageId, pageNumber, getSettings().getPageSecurity());
		if (ObjectChecker.areNotEqual(pageSecurity, DTOSecurityFieldAuthority.NotDefined))
			return pageSecurity;
		return calcPageSecurity(targetEntity, pageId, pageNumber, getFullProfile().getPageSecurity());
	}

	private String calcPageSecurity(String targetEntity, String pageId, int pageNumber, List<DTOPageSecurity> lines)
	{
		if (ObjectChecker.isEmptyOrNull(lines))
			return DTOSecurityFieldAuthority.NotDefined;
		for (DTOPageSecurity authority : lines)
		{
			if ((sameEntity(targetEntity, authority.getEntityType()) || entityTypeListContainsEntity(authority.getEntityTypeList(), targetEntity)
					|| ObjectChecker.areAllEmptyOrNull(authority.getEntityType(), authority.getEntityTypeList())) && (
					ObjectChecker.areEqual(pageId, authority.getPageName()) || ObjectChecker
							.areEqual("" + (pageNumber + 1), authority.getPageName())))
				return authority.getAuthorityType();
		}
		return DTOSecurityFieldAuthority.NotDefined;
	}

	public EntityReferenceData getDefaultShortCuts()
	{
		return defaultShortCuts;
	}

	public void setDefaultShortCuts(EntityReferenceData defaultShortCuts)
	{
		this.defaultShortCuts = defaultShortCuts;
	}

	public EntityReferenceData getEmployee()
	{
		return employee;
	}

	public void setEmployee(EntityReferenceData employee)
	{
		this.employee = employee;
	}

	public DTOUserSettings getSettings()
	{
		return settings;
	}

	public void setSettings(DTOUserSettings settings)
	{
		this.settings = settings;
	}

	public TrueFalseNotSpecified userCanEditField(String targetEntity, String fieldId, ApplicableWhen applicableWhen)
	{
		if (getFullProfile() == null)
			return TrueFalseNotSpecified.False;
		if (ObjectChecker.areAllEmptyOrNull(getSettings().getDisabledFields(), getFullProfile().getDisabledFields()))
			return TrueFalseNotSpecified.NotSpecified;
		if (targetEntity == null || fieldId == null)
			return TrueFalseNotSpecified.True;
		if (ObjectChecker.isTrue(getFullProfile().getFullAuthority()))
			return TrueFalseNotSpecified.True;
		DTOAbstractSecurityFieldAuthority authority = getFieldSecurityLine(targetEntity, fieldId, applicableWhen);
		if (authority == null)
			return TrueFalseNotSpecified.NotSpecified;
		return ObjectChecker.areNotEqual(authority.getAuthorityType(), DTOSecurityFieldAuthority.Disabled) ?
				TrueFalseNotSpecified.True :
				TrueFalseNotSpecified.False;
	}

	public boolean userCanNotCopyLine(String targetEntity, String fieldId, ApplicableWhen applicableWhen)
	{
		return userCanNotOnALineConsiderAlternative(targetEntity, fieldId, applicableWhen, l -> l.getPreventCopyLine());
	}

	public boolean userCanNotInsertLine(String targetEntity, String fieldId, ApplicableWhen applicableWhen)
	{
		return userCanNotOnALineConsiderAlternative(targetEntity, fieldId, applicableWhen, l -> l.getPreventInsertLine());
	}

	public boolean userCanNotDeleteLine(String targetEntity, String fieldId, ApplicableWhen applicableWhen)
	{
		return userCanNotOnALineConsiderAlternative(targetEntity, fieldId, applicableWhen, l -> l.getPreventDeleteLine());
	}

	private boolean userCanNotOnALineConsiderAlternative(String targetEntity, String fieldId, ApplicableWhen applicableWhen,
			Function<DTOAbstractSecurityFieldAuthority, Boolean> fetcher)
	{
		TrueFalseNotSpecified userCanNotOnALine = userCanNotOnALine(targetEntity, fieldId, applicableWhen, fetcher);
		if (userCanNotOnALine.isSpecified())
			return userCanNotOnALine.isTrue();
		if (getAlternativeUsers() != null)
			for (DTOUser user : getAlternativeUsers())
			{
				TrueFalseNotSpecified userCanNotOnALine1 = user.userCanNotOnALine(targetEntity, fieldId, applicableWhen, fetcher);
				if (userCanNotOnALine1.isSpecified())
					return userCanNotOnALine.isTrue();
			}
		return false;
	}

	private TrueFalseNotSpecified userCanNotOnALine(String targetEntity, String fieldId, ApplicableWhen applicableWhen,
			Function<DTOAbstractSecurityFieldAuthority, Boolean> fetcher)
	{
		if (getFullProfile() == null)
			return TrueFalseNotSpecified.NotSpecified;
		if (targetEntity == null || fieldId == null)
			return TrueFalseNotSpecified.False;
		if (ObjectChecker.isTrue(getFullProfile().getFullAuthority()))
			return TrueFalseNotSpecified.False;
		if (ObjectChecker.areAllEmptyOrNull(getSettings().getDisabledFields(), getFullProfile().getDisabledFields()))
			return TrueFalseNotSpecified.NotSpecified;
		DTOAbstractSecurityFieldAuthority authority = getFieldSecurityLine(targetEntity, fieldId, applicableWhen);
		if (authority == null)
			return TrueFalseNotSpecified.NotSpecified;
		return TrueFalseNotSpecified.fromBoolean(ObjectChecker.isTrue(fetcher.apply(authority)));
	}

	private boolean sameEntity(String targetEntity, String authorityEntity)
	{
		return ObjectChecker.areEqual(targetEntity, authorityEntity);
	}

	public boolean userCanAction(String entityType, String actionId)
	{
		if (getFullProfile() == null || getFullProfile().getStandardLines() == null)
			return false;
		if (entityType == null)
			return false;
		if (getFullProfile().getFullAuthority() || ObjectChecker.isEmptyOrNull(actionId))
			return true;
		DTOActionSecurityLine line = getActionSecurityLine(entityType, actionId);
		if (line != null)
		{
			return ObjectChecker.areEqual(line.getAuthorityType(), "enabled");
		}
		return true;
	}

	private DTOActionSecurityLine getActionSecurityLine(String targetEntity, String actionId)
	{
		DTOActionSecurityLine wildCardLine = null;
		for (DTOActionSecurityLine authority : getFullProfile().getActionLines())
		{
			if (sameEntity(targetEntity, authority.getTargetEntity()))
			{
				if (ObjectChecker.areEqual(actionId, authority.getActionId()))
					return authority;
				if (wildCardLine == null && ObjectChecker.areEqual("*", authority.getActionId()))
					wildCardLine = authority;
			}

		}
		for (DTOActionSecurityLine authority : getFullProfile().getActionLines())
		{
			if (entityTypeListContainsEntity(authority.getTargetEntities(), targetEntity))
			{
				if (ObjectChecker.areEqual(actionId, authority.getActionId()))
					return authority;
				if (wildCardLine == null && ObjectChecker.areEqual("*", authority.getActionId()))
					wildCardLine = authority;
			}
		}
		if (wildCardLine == null)
		{
			for (DTOActionSecurityLine authority : getFullProfile().getActionLines())
			{
				if (ObjectChecker.areAllEmptyOrNull(authority.getTargetEntities(), authority.getTargetEntity()))
				{
					if (ObjectChecker.areEqual(actionId, authority.getActionId()))
						return authority;
					if (wildCardLine == null && ObjectChecker.areEqual("*", authority.getActionId()))
						wildCardLine = authority;
				}
			}
		}
		return wildCardLine;
	}

	public List<DTOUser> getAlternativeUsers()
	{
		if (alternativeUsers == null)
			alternativeUsers = new ArrayList<>();
		return alternativeUsers;
	}

	public void setAlternativeUsers(List<DTOUser> alternativeUsers)
	{
		this.alternativeUsers = alternativeUsers;
	}

	public EntityReferenceData fetchUsersCounter()
	{
		if (getSettings() != null && getSettings().getUsersCounter() != null)
			return getSettings().getUsersCounter();
		if (getFullProfile() != null && getFullProfile().getUsersCounter() != null)
			return getFullProfile().getUsersCounter();
		return null;
	}
}
