package com.namasoft.common.flatobjects;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;

import com.namasoft.common.constants.DateWrapper;
import com.namasoft.common.utilities.CollectionsUtility;
import com.namasoft.common.utilities.NaMaLogger;
import com.namasoft.common.utilities.ObjectChecker;

@SuppressWarnings({ "rawtypes", "unchecked" })
public class CommonFlatObjectUtils
{
	private static Class[] NaMaPrimitives = { DateWrapper.class, Date.class, BigDecimal.class, BigInteger.class };
	public static boolean debug = false;

	public static boolean isNaMaPrimitive(Class returnType)
	{
		if (returnType.isPrimitive() || returnType.isArray() || returnType.isEnum())
			return true;
		try
		{
		}
		catch (Throwable e)
		{
			System.err.println(returnType.getName());
		}
		if (returnType.getName().startsWith("java.lang."))
			return true;
		for (int i = 0; i < NaMaPrimitives.length; i++)
		{
			if (returnType.equals(NaMaPrimitives[i]))
				return true;
		}
		return false;
	}

	public static <T> T clone(T value)
	{
		if (value == null)
			return null;
		if (value instanceof FlatObject)
			return (T) ((FlatObject) value).clone();
		if (value instanceof FlatObjectList)
			return (T) ((FlatObjectList) value).clone();
		if (value instanceof DateWrapper)
			return (T) ((DateWrapper) value).clone();
		if (value instanceof Date)
			return (T) ((Date) value).clone();
		if (value instanceof FlatObjectPrimitive)
			return (T) ((FlatObjectPrimitive) value).clone1();
		if (isNaMaPrimitive(value.getClass()))
			return value;
		throw new RuntimeException("Could not clone object:" + value + " of type: " + value.getClass());
	}

	public static boolean areEqual(FlatObject object, FlatObject clone, String... ignoreFields)
	{
		if (ignoreFields.length == 0)
			return ObjectChecker.areEqual(object, clone);

		if (object == clone)
			return true;
		if (object == null)
		{
			if (debug)
				NaMaLogger.error("Equality Check: Null Object");
			return false;
		}
		if (clone == null)
		{
			if (debug)
				NaMaLogger.error("Equality Check: Null Clone");
			return false;
		}

		HashMap<String, Object> objectFlat = flatten(object.getValuesMap(), new HashMap<String, Object>(), "", "");
		HashMap<String, Object> cloneFlat = flatten(clone.getValuesMap(), new HashMap<String, Object>(), "", "");

		List<String> objectKeys = new ArrayList<String>(objectFlat.keySet());
		List<String> cloneKeys = new ArrayList<String>(cloneFlat.keySet());
		List<String> keys = CollectionsUtility.join(objectKeys, cloneKeys);
		List<String> ignore = new ArrayList<String>(Arrays.asList(ignoreFields));
		Collections.sort(keys);
		for (int i = keys.size() - 1; i >= 1; i--)
		{
			if (keys.get(i).equals(keys.get(i - 1)))
				keys.remove(i);
		}
		Collections.sort(ignore);
		removeIgnoredFields(objectFlat, cloneFlat, keys, ignore);
		if (!areSizesEqual(objectFlat, cloneFlat))
		{
			if (debug)
				NaMaLogger.error("Equality Check: Unequal sizes");
			return false;
		}
		return checkEquality(cloneFlat, objectFlat);
	}

	private static boolean checkEquality(HashMap<String, Object> cloneFlat, HashMap<String, Object> objectFlat)
	{
		Iterator<Entry<String, Object>> i = objectFlat.entrySet().iterator();
		while (i.hasNext())
		{
			Entry<String, Object> e = i.next();
			String key = e.getKey();
			Object objectValue = e.getValue();
			Object cloneValue = cloneFlat.get(key);
			if (ObjectChecker.areNotEqual(objectValue, cloneValue))
			{
				if (debug)
					NaMaLogger.error("Equality Check: Mismatch found in key: {0}, objectValue: {1}, cloneValue: {2}", key, objectValue, cloneValue);
				return false;
			}
		}
		return true;
	}

	private static boolean areSizesEqual(HashMap<String, Object> objectFlat, HashMap<String, Object> cloneFlat)
	{
		if (objectFlat.size() != cloneFlat.size())
		{
			if (debug)
			{
				checkEquality(cloneFlat, objectFlat);
			}
			return false;
		}
		return true;
	}

	private static void removeIgnoredFields(HashMap<String, Object> objectFlat, HashMap<String, Object> cloneFlat, List<String> keys,
			List<String> ignore)
	{
		int lastKeyIndex = 0;
		for (int i = 0; i < ignore.size(); i++)
		{
			boolean foundKeyStartsWithIgnore = false;
			for (int j = lastKeyIndex; j < keys.size(); j++)
			{
				if (keys.get(j).startsWith(ignore.get(i)))
				{
					objectFlat.remove(keys.get(j));
					cloneFlat.remove(keys.get(j));
					if (!foundKeyStartsWithIgnore)
					{
						foundKeyStartsWithIgnore = true;
						lastKeyIndex = j;
					}
				}
				else
				{
					if (foundKeyStartsWithIgnore)
						// if this key does not start with the ignore field
						// and there was a key before this one starts with
						// the ignore field, all consecutive keys will not
						// because keys are sorted
						break;
				}
			}
		}
	}

	public static HashMap<String, Object> flatten(HashMap<String, Object> current, HashMap<String, Object> container, String prefix, String suffix)
	{
		if (ObjectChecker.isEmptyOrNull(current))
			return container;
		for (Entry<String, Object> entry : current.entrySet())
		{
			Object value = entry.getValue();
			String key = entry.getKey();
			flattenValue(container, prefix, suffix, value, key);
		}
		return container;
	}

	private static void flattenValue(HashMap<String, Object> container, String prefix, String suffix, Object value, String key)
	{
		if (ObjectChecker.isNotEmptyOrNull(value))
		{
			if (value instanceof FlatObject)
			{
				flatten(((FlatObject) value).getValuesMap(), container, prefix + key + ".", suffix);
			}
			else if (value instanceof FlatObjectList)
			{
				FlatObjectList list = (FlatObjectList) value;
				for (int i = 0; i < list.getList().size(); i++)
				{
					flattenValue(container, prefix, suffix + "[" + key + "-" + i + "]", list.getList().get(i), key);
				}
			}
			else
			{
				container.put(prefix + key + suffix, value);
			}
		}
	}
}
