package com.namasoft.common.hijri;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlRootElement;

import com.namasoft.common.NaMaDTO;
import com.namasoft.common.utilities.CollectionsUtility;
import com.namasoft.common.utilities.ObjectChecker;
import com.namasoft.common.exceptions.NaMaServiceExcepption;

@SuppressWarnings({ "serial", "deprecation" })
@XmlAccessorType(XmlAccessType.PROPERTY)
@XmlRootElement
public class CmnHijrTable extends NaMaDTO
{
	private Date gregStart;
	private int hijriStartYear;
	private List<Integer> monthLengthes;
	private List<Integer> monthDaysTotals;

	public CmnHijrTable()
	{
	}

	public CmnHijrTable(Date gregStart, int hijriStartYear, List<Integer> monthLengthes)
	{
		this.gregStart = gregStart;
		this.hijriStartYear = hijriStartYear;
		this.monthLengthes = monthLengthes;
		monthDaysTotals = new ArrayList<Integer>(this.monthLengthes.size());
		monthDaysTotals.add(0);
		for (int i = 0; i < monthLengthes.size(); i++)
			monthDaysTotals.add(CollectionsUtility.getLast(monthDaysTotals) + monthLengthes.get(i));
	}

	public Date toGregorian(HijriDate d)
	{
		int monthDaysTotalsIndex = (d.getY() - hijriStartYear) * 12 + d.getM() - 1;
		if (monthDaysTotalsIndex < 0 || monthDaysTotalsIndex >= monthDaysTotals.size())
			throw new NaMaServiceExcepption("The provided date is before configured hijri calendar");
		int totalDaysToAdd = monthDaysTotals.get(monthDaysTotalsIndex) + d.getD() - 1;
		Date greg = new Date(gregStart.getTime());
		greg.setDate(greg.getDate() + totalDaysToAdd);
		return greg;
	}

	public HijriDate fromGregorian(Date gregorian)
	{
		if (gregorian == null)
			return null;
		if (ObjectChecker.isEmptyOrNull(monthDaysTotals))
			throw new NaMaServiceExcepption("You can not use hijri dates because hijri files are not configured");
		int days = CalendarUtils.getDaysBetween(gregStart, gregorian) + 1;
		int hijriMonthIndex = 0;
		for (int i = 1; i < monthDaysTotals.size(); i++)
		{
			if (days >= monthDaysTotals.get(i - 1) && days <= monthDaysTotals.get(i))
			{
				hijriMonthIndex = i - 1;
				break;
			}
		}
		int y = hijriMonthIndex / 12 + hijriStartYear;
		int m = hijriMonthIndex % 12 + 1;
		int d = days - monthDaysTotals.get(hijriMonthIndex);
		if (hijriMonthIndex == 0 && d > monthDaysTotals.get(1))
			throw new NaMaServiceExcepption("This date exceeds the provided hijri range: " + gregorian.getDate() + "/" + (gregorian.getMonth() + 1)
					+ "/" + (gregorian.getYear() + 1900));
		return new HijriDate(y, m, d);
	}

	public static void main(String[] args)
	{
	}

	public Date getGregStart()
	{
		return gregStart;
	}

	public void setGregStart(Date gregStart)
	{
		this.gregStart = gregStart;
	}

	public int getHijriStartYear()
	{
		return hijriStartYear;
	}

	public void setHijriStartYear(int hijriStartYear)
	{
		this.hijriStartYear = hijriStartYear;
	}

	public List<Integer> getMonthLengthes()
	{
		return monthLengthes;
	}

	public void setMonthLengthes(List<Integer> monthLengthes)
	{
		this.monthLengthes = monthLengthes;
	}

	public List<Integer> getMonthDaysTotals()
	{
		return monthDaysTotals;
	}

	public void setMonthDaysTotals(List<Integer> monthDaysTotals)
	{
		this.monthDaysTotals = monthDaysTotals;
	}

}
