package com.namasoft.common.utils.translation;

import com.namasoft.common.constants.Language;
import com.namasoft.common.utilities.*;
import com.namasoft.common.utils.NaMaLayersConnector;
import com.namasoft.specialserialization.*;

import java.text.MessageFormat;
import java.util.*;
import java.util.Map.Entry;
import java.util.stream.Collectors;

public class AggregatedMessageTranslator implements MessageTranslator
{
	private static List<MessageTranslator> translators = new ArrayList<>();
	private static AggregatedMessageTranslator instance;
	private static List<TranslatorsProvider> translatorProviders;
	private final Map<String, String> arCache = Collections.synchronizedMap(new HashMap<String, String>());
	private final Map<String, String> enCache = Collections.synchronizedMap(new HashMap<String, String>());
	public static boolean warnAboutMissingTranslation = true;
	public static boolean includeGeneratedTranslations = true;

	public static synchronized AggregatedMessageTranslator instance()
	{
		if (instance == null)
		{
			instance = new AggregatedMessageTranslator();
			addTranslators(Arrays.asList(MessageTranslatorFromFile.fromClassPathFiles("text/cmn-ar.properties", "text/cmn-en.properties", 0),
					MessageTranslatorFromFile.fromClassPathFiles("text/cmn-ar-2.properties", "text/cmn-en-2.properties", 0),
					MessageTranslatorFromFile.fromClassPathFiles("text/cmn-ar-gen.properties", "text/cmn-en-gen.properties", 0),
					MessageTranslatorFromFile.fromClassPathFiles("text/cmn-ar-msg.properties", "text/cmn-en-msg.properties", 0),
					MessageTranslatorFromFile.fromClassPathFiles("text/over-ar.properties", "text/over-en.properties", 99),
					MessageTranslatorFromFile.fromClassPathFiles("over-ar.properties", "over-en.properties", 100)));
			if (ObjectChecker.isTrue(System.getProperty("addmfgtranslations")))
				addTranslators(Arrays.asList(MessageTranslatorFromFile.fromClassPathFiles("text/mfg-ar.properties", "text/mfg-en.properties", 10)));
			if (NaMaLayersConnector.getInstance() != null)
				addTranslators(NaMaLayersConnector.getInstance().getDBTranslators());
			Set<Class<? extends TranslatorsProvider>> translatorProviders = ScanningUtils.getSubTypesOf(TranslatorsProvider.class);
			ReflectionUtils.removeAbstractClasses(translatorProviders);
			if (AggregatedMessageTranslator.translatorProviders == null)
				AggregatedMessageTranslator.translatorProviders = new ArrayList<>();
			for (Class<? extends TranslatorsProvider> class1 : translatorProviders)
			{
				try
				{
					AggregatedMessageTranslator.translatorProviders.add(class1.newInstance());
				}
				catch (Exception e)
				{
					throw new RuntimeException(e);
				}
			}
			for (TranslatorsProvider provider : AggregatedMessageTranslator.translatorProviders)
			{
				addTranslators(provider.getTranslators());
			}
			instance.processRecursions(instance);
			instance.processReplacements(translators);
		}
		return instance;
	}

	public static void addTranslators(List<MessageTranslator> toAdd)
	{
		if (ObjectChecker.isEmptyOrNull(toAdd))
			return;
		if (!includeGeneratedTranslations)
			toAdd = toAdd.stream().filter(t -> !(t instanceof MessageTranslatorFromFile fromFile) || !fromFile.isGeneratedFile())
					.collect(Collectors.toUnmodifiableList());
		translators.addAll(toAdd);
		Collections.sort(translators, (o1, o2) -> Long.valueOf(o2.getOrder()).compareTo(o1.getOrder()));
	}

	@Override
	public String translate(Language lang, String message, Object... arguments)
	{
		if (ObjectChecker.isEmptyOrNull(message))
			return "";
		Object[] translatedArguments = translateArguments(lang, arguments);
		String translatedMessage;
		if (getCache(lang).containsKey(message))
		{
			translatedMessage = getCache(lang).get(message);
			return formatIfNeeded(translatedMessage, translatedArguments);
		}
		else
		{
			translatedMessage = translateFromTranslators(lang, message);
			if (translatedMessage != null)
			{
				getCache(lang).put(message, translatedMessage);
				return formatIfNeeded(translatedMessage, translatedArguments);
			}
		}
		String modifiedMessage = message;
		int firstDotIndex = modifiedMessage.indexOf('.');
		while (firstDotIndex >= 0)
		{
			modifiedMessage = modifiedMessage.substring(firstDotIndex + 1);
			String modifiedMessageTranslation = translateFromTranslators(lang, modifiedMessage);
			if (modifiedMessageTranslation != null)
			{
				getCache(lang).put(message, modifiedMessageTranslation);
				return formatIfNeeded(modifiedMessageTranslation, translatedArguments);
			}
			firstDotIndex = modifiedMessage.indexOf('.');
		}
		if (ObjectChecker.isNotEmptyOrNull(message) && warnAboutMissingTranslation)
			NaMaLogger.warn("Can not find key {0} for language {1}", message, lang);
		getCache(lang).put(message, message);
		return formatIfNeeded(message, translatedArguments);
	}

	private static String formatIfNeeded(String message, Object[] translatedArguments)
	{
		if (translatedArguments.length > 0)
			return MessageFormat.format(message, translatedArguments);
		else
			return message;
	}

	private Map<String, String> getCache(Language lang)
	{
		return lang == Language.English ? enCache : arCache;
	}

	private String translateFromTranslators(Language lang, String message)
	{
		return translateFromTranslators(lang, message, null);
	}

	public String translateFromTranslators(Language lang, String message, MessageTranslator except)
	{
		for (MessageTranslator translator : translators)
		{
			if (except == translator)
				continue;
			String translatedMessage = translator.translate(lang, message);
			if (translatedMessage != null)
				return translatedMessage;
		}
		return null;
	}

	@Override
	public List<String> getKeys()
	{
		List<String> keys = new ArrayList<>();
		for (MessageTranslator translator : translators)
		{
			keys.addAll(translator.getKeys());
		}
		return keys;
	}

	private Object[] translateArguments(Language lang, Object[] arguments)
	{
		Object[] translatedArguments = new Object[arguments.length];
		for (int i = 0; i < arguments.length; i++)
		{
			translatedArguments[i] = TranslationUtil.translateArgument(lang, arguments[i]);
		}
		return translatedArguments;
	}

	@Override
	public long getOrder()
	{
		return 0;
	}

	@Override
	public void processRecursions(AggregatedMessageTranslator aggregatedTranslator)
	{
		for (MessageTranslator translator : translators)
		{
			translator.processRecursions(this);
		}
	}

	@Override
	public void replace(String key, String value)
	{
		for (Entry<String, String> entry : arCache.entrySet())
		{
			entry.setValue(entry.getValue().replace(key, value));
		}
		for (Entry<String, String> entry : enCache.entrySet())
		{
			entry.setValue(entry.getValue().replace(key, value));
		}
	}

	@Override
	public void processReplacements(List<MessageTranslator> translators)
	{
		// reversed order
		for (int i = translators.size() - 1; i >= 0; i--)
		{
			MessageTranslator translator = translators.get(i);
			translator.processReplacements(translators);
		}
	}

	public static void reset()
	{
		instance = null;
		translators = new ArrayList<>();
	}

}
