package com.namasoft.taxauthority;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.namasoft.common.utilities.*;
import com.namasoft.taxauthority.ereceipt.*;
import com.namasoft.taxauthority.jordan.*;
import com.namasoft.taxauthority.ksa.entities.*;
import com.namasoft.taxauthority.ublinvoice.UBLAbsEInvoice;
import jakarta.xml.bind.annotation.*;

import java.math.*;
import java.util.*;
import java.util.stream.Collectors;

@XmlSeeAlso({ TaxAuthorityInvoice.class, TaxAuthorityDebitNote.class, TaxAuthorityCreditNote.class, TaxAuthorityCancelDocument.class, EReceipt.class,
		EReturnReceipt.class, ZATCAAbsEInvoice.class, ZATCAECreditNote.class, ZATCAEDebitNote.class, ZATCAEInvoice.class, AbsJoFotaraInvoice.class,
		UBLAbsEInvoice.class, JoFotaraInvoice.class, JoFotaraInvoiceReturn.class, TaxAuthorityExportInvoice.class, TaxAuthorityExportDebitNote.class,
		TaxAuthorityExportCreditNote.class })
public abstract class EInvoiceDoc
{
	@JsonIgnore
	private Object origin;
	@JsonIgnore
	private Object cancelledDocument;
	@JsonIgnore
	private boolean personReceiver;
	@JsonIgnore
	private boolean cashPaymentMethod;
	private List<TaxAuthorityTotalTax> taxTotals;

	@XmlTransient
	public Object getOrigin()
	{
		return origin;
	}

	public void setOrigin(Object origin)
	{
		this.origin = origin;
	}

	public boolean isPersonReceiver()
	{
		return personReceiver;
	}

	public void setPersonReceiver(boolean personReceiver)
	{
		this.personReceiver = personReceiver;
	}

	public boolean isCashPaymentMethod()
	{
		return cashPaymentMethod;
	}

	public void setCashPaymentMethod(boolean cashPaymentMethod)
	{
		this.cashPaymentMethod = cashPaymentMethod;
	}

	public List<TaxAuthorityTotalTax> getTaxTotals()
	{
		return taxTotals;
	}

	public void setTaxTotals(List<TaxAuthorityTotalTax> taxTotals)
	{
		this.taxTotals = taxTotals;
	}

	public boolean createSignature()
	{
		return true;
	}

	public abstract void updateDocumentVersion(boolean signed);

	public void updateReferences(List<String> references)
	{

	}

	public abstract void updateTaxpayerActivityCode(String activityCode);

	public abstract void updateDateTimeIssued(String dateTimeIssued);

	public void updateServiceDeliveryDate(String serviceDeliveryDate)
	{

	}

	public abstract String fetchDateTimeIssued();

	public abstract void updateIssuer(TaxAuthorityEntityWithAddress issuer);

	public abstract TaxAuthorityEntityWithAddress fetchIssuer();

	public abstract void updateDelivery(TaxAuthorityDelivery delivery);

	public abstract TaxAuthorityDelivery fetchDelivery();

	public abstract void updateReceiver(TaxAuthorityReceiver receiver);

	public abstract TaxAuthorityReceiver fetchReceiver();

	public abstract void updatePayment(TaxAuthorityPayment payment);

	public abstract TaxAuthorityPayment fetchPayment();

	public void updatePurchaseOrderReference(String purchaseOrderReference)
	{
	}

	public void updatePurchaseOrderDescription(String purchaseOrderDescription)
	{
	}

	public void updateSalesOrderReference(String salesOrderReference)
	{
	}

	public void updateSalesOrderDescription(String salesOrderDescription)
	{
	}

	public String fetchSalesOrderDescription()
	{
		return null;
	}

	public boolean shouldUpdateSalesOrderDescriptionByNetTotal()
	{
		if (ObjectChecker.isNotEmptyOrNull(fetchSalesOrderDescription()))
			return false;
		return ublInvoice();
	}

	public boolean ublInvoice()
	{
		return false;
	}

	public void updateProformaInvoiceNumber(String proformaInvoiceNumber)
	{
	}

	public void updateTotals()
	{
		List<? extends EInvoiceDocLine> lines = fetchLines();
		setTaxTotals(taxTotals(lines));
		updateTotalSalesAmount(NaMaMath.totalizeDecimalStream(lines.stream().map(EInvoiceDocLine::fetchSalesTotal)));
		updateTotalDiscountAmount(NaMaMath.totalizeDecimalStream(lines.stream().map(l -> NaMaMath.zeroIfNull(l.fetchDiscount().getAmount()))));
		updateNetAmount(NaMaMath.totalizeDecimalStream(lines.stream().map(EInvoiceDocLine::fetchNetTotal)));
		updateExtraDiscountAmount(BigDecimal.ZERO);
		updateTotalItemsDiscountAmount(
				NaMaMath.totalizeDecimalStream(lines.stream().map(EInvoiceDocLine::fetchItemsDiscount).map(NaMaMath::zeroIfNull)));
		updateFinalTotalAmount(
				NaMaMath.totalizeDecimalStream(lines.stream().map(EInvoiceDocLine::fetchFinalTotal)).subtract(fetchExtraDiscountAmount()));
	}

	public void roundValues(int scale)
	{
		updateTotalSalesAmount(NaMaMath.zeroIfNull(fetchTotalSalesAmount()).setScale(scale, RoundingMode.HALF_UP));
		updateTotalDiscountAmount(NaMaMath.zeroIfNull(fetchTotalDiscountAmount()).setScale(scale, RoundingMode.HALF_UP));
		updateNetAmount(NaMaMath.zeroIfNull(fetchNetAmount()).setScale(scale, RoundingMode.HALF_UP));
		updateExtraDiscountAmount(NaMaMath.zeroIfNull(fetchExtraDiscountAmount()).setScale(scale, RoundingMode.HALF_UP));
		updateTotalItemsDiscountAmount(NaMaMath.zeroIfNull(fetchTotalItemsDiscountAmount()).setScale(scale, RoundingMode.HALF_UP));
		updateFinalTotalAmount(NaMaMath.zeroIfNull(fetchFinalTotalAmount()).setScale(scale, RoundingMode.HALF_UP));
	}

	public abstract void updateTotalSalesAmount(BigDecimal totalSalesAmount);

	public abstract void updateTotalDiscountAmount(BigDecimal totalDiscountAmount);

	public abstract BigDecimal fetchTotalDiscountAmount();

	public abstract void updateNetAmount(BigDecimal netAmount);

	public abstract void updateExtraDiscountAmount(BigDecimal extraDiscountAmount);

	public abstract void updateTotalItemsDiscountAmount(BigDecimal totalItemsDiscountAmount);

	public abstract BigDecimal fetchTotalItemsDiscountAmount();

	public BigDecimal fetchExtraDiscountAmount()
	{
		return BigDecimal.ZERO;
	}

	public List<TaxAuthorityTotalTax> taxTotals(List<? extends EInvoiceDocLine> details)
	{
		List<TaxAuthorityLineTax> lines = details.stream().map(EInvoiceDocLine::fetchTaxableItems).flatMap(Collection::stream)
				.collect(Collectors.toList());
		List<TaxAuthorityTotalTax> taxes = new ArrayList<>();
		lines.forEach(l -> TaxAuthorityTotalTax.addAmountToTotalIfNeeded(taxes, l));
		return taxes;
	}

	public abstract void updateInternalCode(String internalCode);

	public abstract String fetchInternalCode();

	public abstract <T extends EInvoiceDocLine> void addLine(T taxInvoiceLine);

	public boolean shouldAddSignature()
	{
		if (ObjectChecker.isFalse(createSignature()))
			return false;
		return documentVersionRequireSignature();
	}

	public abstract boolean documentVersionRequireSignature();
	public abstract boolean shouldValidateSignature();

	public List<TaxAuthoritySignature> signatures()
	{
		if (getSignatures() == null)
			setSignatures(new ArrayList<>());
		return getSignatures();
	}

	public abstract List<TaxAuthoritySignature> getSignatures();

	public abstract void setSignatures(List<TaxAuthoritySignature> signatures);

	public void addSignature(TaxAuthoritySignature signature)
	{
		signatures().add(signature);
	}

	public boolean cancelDocument()
	{
		return false;
	}

	public boolean notCancelDocument()
	{
		return !cancelDocument();
	}

	public String cancelReason()
	{
		return null;
	}

	@XmlTransient
	public Object getCancelledDocument()
	{
		return cancelledDocument;
	}

	public void setCancelledDocument(Object cancelledDocument)
	{
		this.cancelledDocument = cancelledDocument;
	}

	public boolean receiverIdRequired(BigDecimal min)
	{
		if (exportDocument())
			return true;
		if (ObjectChecker.areNotEqual(fetchReceiver().getType(), "P"))
			return true;
		if (ObjectChecker.isEmptyOrZero(min))
			return true;
		BigDecimal total = NaMaMath.zeroIfNull(fetchFinalTotalAmount());
		if (ObjectChecker.isEmptyOrZero(total))
			return false;
		return total.compareTo(min) > 0;
	}

	public boolean receiverNameIsRequired(BigDecimal min)
	{
		if (exportDocument())
			return true;
		if (ObjectChecker.areEqual(fetchReceiver().getType(), "B"))
			return true;
		if (ObjectChecker.areNotEqual(fetchReceiver().getType(), 'P'))
			return false;
		if (ObjectChecker.isEmptyOrZero(min))
			return true;
		BigDecimal total = NaMaMath.zeroIfNull(fetchFinalTotalAmount());
		if (ObjectChecker.isEmptyOrZero(total))
			return false;
		return total.compareTo(min) > 0;
	}

	public abstract BigDecimal fetchFinalTotalAmount();

	public abstract void updateFinalTotalAmount(BigDecimal totalAmount);

	public abstract void updateLines(List<? extends EInvoiceDocLine> lines);

	public abstract <T extends EInvoiceDocLine> List<T> fetchLines();

	public String cancelledDocumentUUID()
	{
		return null;
	}

	public String fetchUUID()
	{
		return null;
	}

	public void updateGeneratedUUID(String uuid)
	{

	}

	public void updateUUID(String uuid)
	{

	}

	public void updateOldUUID(String uuid)
	{

	}

	public void updatePreviousUUID(String uuid)
	{

	}

	public void updateReferenceUUID(String uuid)
	{

	}

	public String fetchReferenceUUID()
	{
		return null;
	}

	public void updatePaymentMethod(String paymentMethodCode)
	{

	}

	public boolean eInvoiceSubmission()
	{
		return true;
	}

	public boolean eReceiptSubmission()
	{
		return !eInvoiceSubmission();
	}

	public abstract BigDecimal fetchTotalSalesAmount();

	public abstract BigDecimal fetchNetAmount();

	public TaxAuthorityTotalTax taxTotalLine(int index)
	{
		if (ObjectChecker.isEmptyOrNull(getTaxTotals()))
			return null;
		if (index >= getTaxTotals().size())
			return null;
		return getTaxTotals().get(index);
	}

	public void updateHeaderCurrency(String currency)
	{

	}

	public void updateHeaderCurrencyRate(BigDecimal rate)
	{

	}

	public String fetchHeaderCurrency()
	{
		return null;
	}

	public BigDecimal fetchHeaderCurrencyRate()
	{
		return BigDecimal.ZERO;
	}

	public abstract String edocumentType();

	public boolean ifReturn()
	{
		return false;
	}

	public void updateDocumentCurrency(String code)
	{

	}

	public void updateTaxCurrencyCode(String code)
	{

	}

	public void updateInvoiceTransactionCode()
	{

	}

	public void updatePreviousInvoiceHash(String previousInvoiceHash)
	{

	}

	public abstract EInvoiceCountry invoiceCountry();

	public abstract boolean hasDefaultTaxCurrency();

	public abstract String defaultCurrency();

	public String defaultTaxCurrency()
	{
		return null;
	}

	public boolean shouldCheckReceiverAddress()
	{
		return true;
	}

	public int taxScale()
	{
		return 5;
	}

	public boolean exportDocument()
	{
		return false;
	}

	public abstract  <T extends EInvoiceDocLine> T createLine();

	public TaxAuthorityEntityWithAddress createIssuer()
	{
		return new TaxAuthorityIssuer();
	}

	public String fetchDebitOrCreditNote()
	{
		return null;
	}

	public boolean debitOrCreditNoteIsRequired()
	{
		return false;
	}

	public String fetchPaymentMethod()
	{
		return null;
	}

	public void updateDebitOrCreditNote(String debitOrCreditNote)
	{
	}

	public void updateInvoiceHeaderCharges(BigDecimal invoiceHeaderCharges)
	{

	}

	public abstract String cashPaymentMethodCode();

	public abstract String otherPaymentMethodCode();
}
