package com.namasoft.common.utilities;

import com.namasoft.common.flatobjects.CommonFlatObjectUtils;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.*;
import java.util.Map.Entry;
import java.util.function.*;

@SuppressWarnings({ "unchecked", "rawtypes" })
public class ObjectChecker
{
	public static boolean areAllEmptyOrNull(Object... objects)
	{
		for (Object o : objects)
		{
			if (isNotEmptyOrNull(o))
				return false;
		}
		return true;
	}

	public static boolean areAllEmptyOrZero(BigDecimal... objects)
	{
		for (BigDecimal o : objects)
		{
			if (isNotEmptyOrZero(o))
				return false;
		}
		return true;
	}

	public static boolean areAllEmptyOrZero(Double... objects)
	{
		for (Double o : objects)
		{
			if (isNotEmptyOrZero(o))
				return false;
		}
		return true;
	}

	public static boolean areAllEmptyOrZero(Long... objects)
	{
		for (Long o : objects)
		{
			if (isNotEmptyOrZero(o))
				return false;
		}
		return true;
	}

	public static boolean isBlankOrNull(String val)
	{
		return val == null || val.isBlank();
	}

	public static boolean isEmptyOrNull(Object object)
	{
		if (object == null)
			return true;
		if (object instanceof NamaObject)
			return ((NamaObject) object).isEmpty();
		if (object instanceof Collection)
			return ((Collection<?>) object).isEmpty();
		if (object instanceof Map)
			return ((Map) object).isEmpty();
		if (object instanceof String)
			return ((String) object).isEmpty();
		if (object instanceof StringBuilder)
			return ((StringBuilder) object).length() == 0;
		return false;
	}

	public static boolean isNotEmptyOrNull(Object object)
	{
		return !isEmptyOrNull(object);
	}

	public static <T> Predicate<T> equalsPredicate(Object object)
	{
		return (o) -> areEqual(o, object);
	}

	public static <T, V> Predicate<T> equalsPredicate(V object, Function<T, V> supplier)
	{
		return (o) -> areEqual(supplier.apply(o), object);
	}

	public static <V> Predicate<V> notEqualPredicate(V object)
	{
		return notEqualPredicate(object, v -> v);
	}

	public static <T, V> Predicate<T> notEqualPredicate(V object, Function<T, V> supplier)
	{
		return (o) -> areNotEqual(supplier.apply(o), object);
	}

	public static boolean areEqual(Object object1, Object object2)
	{
		if (object1 == object2)
			return true;
		if (object1 == null)
			return false;
		if (object2 == null)
			return false;
		if (object1 instanceof Enum)
			object1 = object1.toString();
		if (object2 instanceof Enum)
			object2 = object2.toString();
		if (object1 instanceof BigDecimal && object2 instanceof BigDecimal)
			return ((BigDecimal) object1).compareTo((BigDecimal) object2) == 0;
		return object1.equals(object2);
	}

	public static boolean areNotEqual(Object object1, Object object2)
	{
		return !areEqual(object1, object2);
	}

	public static boolean areAllNotEmptyOrNull(Object... objects)
	{
		for (Object o : objects)
		{
			if (isEmptyOrNull(o))
				return false;
		}
		return true;
	}

	public static boolean areAllNotEmptyOrZero(BigDecimal... objects)
	{
		for (BigDecimal o : objects)
		{
			if (isEmptyOrZero(o))
				return false;
		}
		return true;
	}

	public static boolean areAllNotEmptyOrZero(Integer... objects)
	{
		for (Integer o : objects)
		{
			if (isEmptyOrZero(o))
				return false;
		}
		return true;
	}

	public static boolean areAllEmptyOrZero(Integer... objects)
	{
		for (Integer o : objects)
		{
			if (isNotEmptyOrZero(o))
				return false;
		}
		return true;
	}

	public static <T> int compare(Comparable<T> o1, T o2)
	{
		if (o1 == null && o2 == null)
			return 0;
		if (o1 == null)
			return 1;
		if (o2 == null)
			return -1;
		return o1.compareTo(o2);
	}

	public static boolean isAnyEmptyOrNull(Object... objects)
	{
		for (Object o : objects)
		{
			if (isEmptyOrNull(o))
				return true;
		}
		return false;
	}

	public static boolean isAnyNotEmptyOrNull(Object... objects)
	{
		for (Object o : objects)
		{
			if (isNotEmptyOrNull(o))
				return true;
		}
		return false;
	}

	public static <T> T getFirstNotEmptyObj(List<T> list)
	{
		for (T obj : list)
			if (isNotEmptyOrNull(obj))
				return obj;
		return null;
	}

	@SafeVarargs
	public static <T> T getFirstNotEmptyObj(T... objects)
	{
		for (T obj : objects)
			if (isNotEmptyOrNull(obj))
				return obj;
		return getFirstNotNullObj(objects);
	}

	public static <T> T getFirstNotNullObj(T... objects)
	{
		for (T obj : objects)
			if (obj != null)
				return obj;
		return null;
	}

	public static String toStringOrEmpty(Object object)
	{
		return object == null ? "" : object.toString();
	}

	public static boolean areEqualIgnoringEmptyLines(List<Object> thisList, List<Object> otherList)
	{
		if (ObjectChecker.isEmptyOrNull(thisList) && ObjectChecker.isEmptyOrNull(otherList))
		{
			return true;
		}
		if (thisList == otherList)
			return true;
		if (sizeIgnoringEmptyLines(thisList) != sizeIgnoringEmptyLines(otherList))
			return false;
		int j = 0;
		for (int i = 0; i < thisList.size(); i++)
		{

			while (i < thisList.size() && ObjectChecker.isEmptyOrNull(thisList.get(i)))
			{
				i++;

			}
			while (j < otherList.size() && ObjectChecker.isEmptyOrNull(otherList.get(j)))
			{
				j++;
			}
			boolean iListFinshed = (i >= thisList.size());
			boolean jListFinshed = (j >= otherList.size());
			if (iListFinshed != jListFinshed)
			{
				return false;
			}
			if (iListFinshed && jListFinshed)
				return true;
			if (ObjectChecker.areNotEqual(thisList.get(i), otherList.get(j)))
				return false;
			j++;
		}
		return true;
	}

	public static boolean areEqualIgnoringEmptyLines(HashMap thisMap, HashMap otherMap)
	{
		if (thisMap == otherMap)
			return true;

		if (sizeIgnoringEmptyLines(otherMap) != sizeIgnoringEmptyLines(thisMap))
		{
			if (CommonFlatObjectUtils.debug)
			{
				NaMaLogger.error("Equality Check: sizes not equal");
			}
			return false;
		}

		try
		{
			Iterator<Entry> i = thisMap.entrySet().iterator();
			while (i.hasNext())
			{
				Entry e = i.next();
				Object key = e.getKey();
				Object value = e.getValue();
				if (isEmptyOrNull(value))
					continue;
				if (areNotEqual(value, otherMap.get(key)))
				{
					if (CommonFlatObjectUtils.debug)
						NaMaLogger.error("Equality Check: unequal- key {0}\n value\n {1}\nothervalue\n {2}", e.getKey(), e.getValue(),
								otherMap.get(key));
					return false;
				}
			}
		}
		catch (ClassCastException unused)
		{
			return false;
		}
		catch (NullPointerException unused)
		{
			return false;
		}

		return true;

	}

	public static int sizeIgnoringEmptyLines(HashMap valuesMap)
	{
		return sizeIgnoringEmptyLines(valuesMap.values());
	}

	public static int sizeIgnoringEmptyLines(Iterable<?> iterable)
	{
		int size = 0;
		for (Object value : iterable)
		{
			if (isNotEmptyOrNull(value))
				size++;
		}
		return size;
	}

	public static boolean isTrue_b(Boolean b)
	{
		return isTrue(b);
	}

	public static boolean isTrue(Boolean b)
	{
		return Boolean.TRUE.equals(b);
	}

	public static boolean isTrueOrNull(Boolean b)
	{
		return Boolean.TRUE.equals(b) || b == null;
	}

	public static boolean isFalse(Boolean b)
	{
		return isFalseOrNull(b);
	}

	public static boolean isFalseOrNull(Boolean b)
	{
		return Boolean.FALSE.equals(b) || b == null;
	}

	public static boolean isAnyFalseOrNull(Boolean... bs)
	{
		for (Boolean b : bs)
			if (isFalseOrNull(b))
				return true;

		return false;
	}

	@SafeVarargs
	public static <T> boolean isAnyEqualToFirst(T first, T... others)
	{
		for (Object other : others)
		{
			if (ObjectChecker.areEqual(first, other))
				return true;
		}
		return false;
	}

	public static boolean arePairsEqual(Object... pairs)
	{
		assert pairs.length != 0 && pairs.length % 2 == 0 : "arePairsEqual should have even number of arguments";
		for (int i = 0; i < pairs.length; i += 2)
			if (ObjectChecker.areNotEqual(pairs[i], pairs[i + 1]))
				return false;
		return true;
	}

	public static boolean arePairsNotEqual(Object... pairs)
	{
		assert pairs.length != 0 && pairs.length % 2 == 0 : "arePairsNotEqual should have even number of arguments";
		for (int i = 0; i < pairs.length; i += 2)
			if (ObjectChecker.areEqual(pairs[i], pairs[i + 1]))
				return false;
		return true;
	}

	public static boolean isNotZero(BigDecimal value)
	{
		return !isZero(value);
	}

	public static boolean isZero(BigDecimal value)
	{
		return BigDecimal.ZERO.compareTo(value) == 0;
	}

	public static boolean isZero(Integer value)
	{
		return value == null || value == 0;
	}

	public static boolean isZero(Double value)
	{
		return value == null || value == 0.0;
	}

	public static boolean isNotNegative(BigDecimal value)
	{
		return !isNegative(value);
	}

	public static boolean isNotNegative(Integer value)
	{
		return !isNegative(value);
	}

	public static boolean isNegative(BigDecimal value)
	{
		return BigDecimal.ZERO.compareTo(value) > 0;
	}

	public static boolean isNegative(Integer value)
	{
		return Integer.valueOf(0).compareTo(value) > 0;
	}

	public static boolean isPositive(BigDecimal value)
	{
		return BigDecimal.ZERO.compareTo(value) < 0;
	}

	public static <T extends Comparable> T getMaxValueWithDefault(T defautValue, T... values)
	{
		if (values == null || values.length == 0)
			return defautValue;
		T maxValue = values[0];
		for (int i = 1; i < values.length; i++)
		{
			if (maxValue == null)
			{
				maxValue = values[i];
			}
			else if (values[i] != null)
			{
				if (values[i].compareTo(maxValue) > 0)
					maxValue = values[i];
			}
		}
		if (maxValue == null)
			return defautValue;
		return maxValue;
	}

	public static <T extends Comparable> T getMinValue(T... values)
	{
		if (values == null || values.length == 0)
			return null;
		T minValue = values[0];
		for (int i = 1; i < values.length; i++)
		{
			if (minValue == null)
			{
				minValue = values[i];
			}
			else if (values[i] != null)
			{
				if (values[i].compareTo(minValue) < 0)
					minValue = values[i];
			}
		}
		return minValue;
	}

	public static Integer tryParseInt(Object value)
	{
		return tryParseInt(ObjectChecker.toStringOrEmpty(value));
	}

	public static Integer tryParseInt(String value)
	{
		if (ObjectChecker.isEmptyOrNull(value))
			return null;
		try
		{
			return new BigDecimal(value).intValue();
		}
		catch (Exception e)
		{
			return null;
		}
	}

	public static Long tryParseLong(Object value)
	{
		return tryParseLong(ObjectChecker.toStringOrEmpty(value));
	}

	public static Long tryParseLong(String value)
	{
		try
		{
			return new BigDecimal(toStringOrEmpty(value)).longValue();
		}
		catch (Exception e)
		{
			return null;
		}
	}

	public static Long toLongValue(Object value)
	{
		try
		{
			if (value instanceof Number)
				return ((Number) value).longValue();
			if (value instanceof String)
				return tryParseLong((String) value);
			return null;
		}
		catch (Exception e)
		{
			return null;
		}
	}

	public static BigDecimal tryParseDecimal(String value)
	{
		try
		{
			return new BigDecimal(value);
		}
		catch (Exception e)
		{
			return null;
		}
	}

	public static Boolean valueOfBoolean(String value)
	{
		if (isEmptyOrNull(value))
			return null;
		return Boolean.parseBoolean(value.trim());
	}

	public static Integer valueOfInteger(String value)
	{
		if (isEmptyOrNull(value))
			return null;
		return Integer.parseInt(value);
	}

	public static boolean isNotEmptyOrZero(BigDecimal value)
	{
		return isNotEmptyOrNull(value) && isNotZero(value);
	}

	public static boolean isNotEmptyOrZero(Double value)
	{
		return isNotEmptyOrNull(value) && isNotZero(value);
	}

	public static boolean isEmptyOrZero(BigDecimal value)
	{
		return isEmptyOrNull(value) || isZero(value);
	}

	public static boolean isEmptyOrZero(Integer value)
	{
		return isEmptyOrNull(value) || isZero(value);
	}

	public static boolean isEmptyOrZero(Double value)
	{
		return isEmptyOrNull(value) || isZero(value);
	}

	public static List<String> toStringList(Object... objects)
	{
		ArrayList<String> list = new ArrayList<String>();
		for (Object object : objects)
		{
			list.add(toStringOrEmpty(object));
		}
		return list;
	}

	public static BigDecimal toZeroIfNull(BigDecimal v)
	{
		if (isEmptyOrNull(v))
			return BigDecimal.ZERO;
		return v;
	}

	public static Integer toZeroIfNull(Integer v)
	{
		if (isEmptyOrNull(v))
			return 0;
		return v;
	}

	public static Long toZeroIfNull(Long v)
	{
		if (isEmptyOrNull(v))
			return 0L;
		return v;
	}

	public static BigDecimal toZeroIfNigative(BigDecimal v)
	{
		if (v.compareTo(BigDecimal.ZERO) < 0)
			return BigDecimal.ZERO;
		return v;
	}

	public static boolean isTrue(String str)
	{
		return isTrue(valueOfBoolean(str));
	}

	public static boolean isTrueOrEmpty(String str)
	{
		Boolean b = valueOfBoolean(str);
		return b == null || isTrue(b);
	}

	public static boolean isFalse_OrNull(String str)
	{
		return isFalseOrNull(valueOfBoolean(str));
	}

	public static boolean isNotEmptyOrZero(Integer value)
	{
		return value != null && value != 0;
	}

	public static boolean isNotEmptyOrZero(Long value)
	{
		return value != null && value != 0;
	}

	public static boolean isEmptyOrZero(Long value)
	{
		return !ObjectChecker.isNotEmptyOrZero(value);
	}

	public static boolean isNotZero(Integer value)
	{
		return !isZero(value);
	}

	public static boolean isNotZero(Double value)
	{
		return !isZero(value);
	}

	public static boolean isAnyTrue(Boolean... booleans)
	{
		for (Boolean boolean1 : booleans)
		{
			if (isTrue(boolean1))
				return true;
		}
		return false;
	}

	public static boolean areAllTrue(Boolean... booleans)
	{
		for (Boolean boolean1 : booleans)
		{
			if (isFalseOrNull(boolean1))
				return false;
		}
		return true;
	}

	public static Date todayIfNull(Date v)
	{
		if (isEmptyOrNull(v))
			return new Date();
		return v;
	}

	public static boolean isNegativeOrZero(BigDecimal value)
	{
		return isEmptyOrZero(value) || isNegative(value);
	}

	public static boolean isNegativeOrZero(Integer value)
	{
		return isEmptyOrZero(value) || isNegative(value);
	}

	public static boolean isAnyNotEmptyOrZero(Integer... ns)
	{
		for (Integer d : ns)
			if (isNotEmptyOrZero(d))
				return true;
		return false;
	}

	public static boolean isAnyNotEmptyOrZero(BigDecimal... ns)
	{
		for (BigDecimal d : ns)
			if (isNotEmptyOrZero(d))
				return true;
		return false;
	}

	public static boolean isAnyEmptyOrZero(BigDecimal... ns)
	{
		for (BigDecimal d : ns)
			if (isEmptyOrZero(d))
				return true;
		return false;
	}

	public static boolean isFirstEmptyOrEqualsSecond(Object first, Object second)
	{
		return isEmptyOrNull(first) || areEqual(first, second);
	}

	public static BigDecimal toZeroIfNegative(BigDecimal decimal)
	{
		if (decimal == null)
			return BigDecimal.ZERO;
		if (decimal.compareTo(BigDecimal.ZERO) < 0)
			return BigDecimal.ZERO;
		return decimal;
	}

	public static <T> boolean NOTisAnyEqualToFirst(T first, T... others)
	{
		return !isAnyEqualToFirst(first, others);
	}

	public static boolean isEmptyOrNegative(BigDecimal value)
	{
		if (isEmptyOrZero(value))
			return true;
		return value.compareTo(BigDecimal.ZERO) < 0;
	}

	public static boolean isNotEmptyOrNegative(BigDecimal value)
	{
		return !isEmptyOrNegative(value);
	}

	public static String toStringOrNull(Object o)
	{
		if (isEmptyOrNull(o))
			return null;
		return o.toString();
	}

	public static String toStringOrNullOnlyForNull(Object o)
	{
		if (o == null)
			return null;
		return o.toString();
	}

	public static BigDecimal toOneIfNull(BigDecimal v)
	{
		if (isEmptyOrNull(v))
			return BigDecimal.ONE;
		return v;
	}

	public static Double getFirstNotEmptyOrZero(Double... ns)
	{
		for (Double n : ns)
			if (isNotEmptyOrZero(n))
				return n;
		return 0.0;
	}

	public static BigDecimal getFirstNotEmptyOrZero(BigDecimal... ns)
	{
		for (BigDecimal n : ns)
			if (isNotEmptyOrZero(n))
				return n;
		return BigDecimal.ZERO;
	}

	public static Integer getFirstNotEmptyOrZero(Integer... ns)
	{
		for (Integer n : ns)
			if (isNotEmptyOrZero(n))
				return n;
		return 0;
	}

	public static <T, U extends Comparable<? super U>> Comparator<T> comparing(Function<? super T, ? extends U> keyExtractor)
	{
		Objects.requireNonNull(keyExtractor);
		return (Comparator<T> & Serializable) (c1, c2) -> {
			U o1 = keyExtractor.apply(c1);
			U o2 = keyExtractor.apply(c2);
			if (o1 == null && o2 == null)
				return 0;
			if (o1 == null)
				return 1;
			if (o2 == null)
				return -1;
			return o1.compareTo(o2);
		};
	}

	public static BigDecimal toOneIfZero(BigDecimal v)
	{
		if (isEmptyOrZero(v))
			return BigDecimal.ONE;
		return v;
	}

	public static boolean isObjectNotEmptyOrZero(Object object)
	{
		return !isObjectEmptyOrZero(object);
	}

	public static Integer toOneIfZero(Integer v)
	{
		if (isEmptyOrZero(v))
			return 1;
		return v;
	}

	public static Long toOneIfZero(Long v)
	{
		if (isEmptyOrZero(v))
			return 1L;
		return v;
	}

	public static boolean isObjectEmptyOrZero(Object object)
	{
		if (ObjectChecker.isEmptyOrNull(object))
			return true;
		if (object instanceof Boolean)
			return isFalseOrNull((Boolean) object);
		if (object instanceof BigDecimal)
			return isEmptyOrZero((BigDecimal) object);
		if (object instanceof Integer)
			return isEmptyOrZero((Integer) object);
		if (object instanceof Double)
			return isEmptyOrZero((Double) object);
		if (object instanceof Number)
			return isEmptyOrZero(new BigDecimal(object.toString()));
		return false;
	}

	public static long toZeroIfNegative(Long l)
	{
		if (l == null)
			return 0;
		if (l < 0)
			return 0;
		return l;
	}

	public static int toZeroIfNegative(Integer l)
	{
		if (l == null)
			return 0;
		if (l < 0)
			return 0;
		return l;
	}

	public static Double toDoubleIfNotNull(BigDecimal value)
	{
		if (value == null)
			return null;
		return value.doubleValue();
	}

	public static BigDecimal toBigDecimalIfNumber(Object value)
	{
		return NaMaMath.zeroIfNull(tryParseDecimal(toStringOrEmpty(value)));
	}

	public static boolean areAllFalseOrNull(Boolean... l)
	{
		for (Boolean b : l)
		{
			if (isTrue(b))
				return false;
		}
		return true;
	}

	public static Long numberToLong(Number number)
	{
		if (number == null)
			return null;
		return number.longValue();
	}

	public static <T> T getFieldFromMap(Map<String, Object> map, String key)
	{
		return (T) map.get(key);
	}

	public static boolean isNotBlankOrNull(String s)
	{
		return !isBlankOrNull(s);
	}

	public static String appendToUrl(String url, String toAppend)
	{
		return (url.endsWith("/") ? url : url + "/") + (toAppend.startsWith("/") ? toAppend.substring(1) : toAppend);
	}
}
