package com.namasoft.common.utils;

import com.namasoft.common.utilities.*;

import java.math.BigDecimal;
import java.text.*;
import java.util.*;

public class PaymentMethodFeeCalculator
{
	public static final String NO_DATE = "ND";
	public static final String NOT_Available = "NA";

	private static class FeesRange
	{
		private BigDecimal fromValue;
		private BigDecimal toValue;
		private BigDecimal feePercentage;
		private BigDecimal feeValue;
		private Date fromDate;
		private Date toDate;
		private BigDecimal maxValue;
		private List<String> forTypes;

		public FeesRange(String fromValue, String toValue, String feePercentage, String feeValue, String fromDate, String toDate, String maxValue,
				List<String> forTypes)
		{
			this.fromValue = new BigDecimal(fromValue);
			this.toValue = new BigDecimal(toValue);
			this.feePercentage = new BigDecimal(feePercentage);
			this.feeValue = new BigDecimal(feeValue);
			this.fromDate = parseDate(fromDate);
			this.toDate = parseDate(toDate);
			this.maxValue = new BigDecimal(maxValue);
			this.forTypes = forTypes;
		}

		static List<String> parseList(String entityTypeList)
		{
			if (ObjectChecker.isEmptyOrNull(entityTypeList) || ObjectChecker.areEqual(entityTypeList, NOT_Available))
				return new ArrayList<>();
			return Arrays.asList(entityTypeList.split("#"));
		}

		private Date parseDate(String date)
		{
			if (ObjectChecker.isEmptyOrNull(date) || ObjectChecker.areEqual(date, NO_DATE))
				return null;
			try
			{
				return new SimpleDateFormat("yyyyMMdd").parse(date);
			}
			catch (ParseException e)
			{
				throw new RuntimeException(e);
			}
		}
	}

	public static BigDecimal calculatePaymentFeesValue(Date valueDate, BigDecimal value, BigDecimal feePercentage, BigDecimal feeValue,
			Integer decimalPlaces, String feeRages, String entityType)
	{
		if (ObjectChecker.isEmptyOrZero(value))
			return BigDecimal.ZERO;
		if (valueDate == null)
			valueDate = new Date();
		if (ObjectChecker.isNotEmptyOrNull(feeRages))
		{
			List<FeesRange> ranges = parse(feeRages);
			for (int i = 0; i < ranges.size(); i++)
			{
				FeesRange r = ranges.get(i);
				if (ObjectChecker.isNotEmptyOrZero(r.fromValue) && value.compareTo(r.fromValue) < 0)
					continue;
				if (ObjectChecker.isNotEmptyOrZero(r.toValue) && value.compareTo(r.toValue) > 0)
					continue;
				if (ObjectChecker.isNotEmptyOrNull(r.fromDate) && valueDate.before(r.fromDate))
					continue;
				if (ObjectChecker.isNotEmptyOrNull(r.toDate) && valueDate.after(r.toDate))
					continue;
				if (ObjectChecker.areAllNotEmptyOrNull(r.forTypes, entityType) && !r.forTypes.contains(entityType))
					continue;
				return calcFeesFromPercentAndValue(value, r.feePercentage, r.feeValue, decimalPlaces, r.maxValue);
			}
		}
		return calcFeesFromPercentAndValue(value, feePercentage, feeValue, decimalPlaces, BigDecimal.ZERO);
	}

	private static List<FeesRange> parse(String feeRages)
	{
		List<String> list = StringUtils.csvLineToList(feeRages);
		List<FeesRange> ranges = new ArrayList<>();
		for (int i = 0; i < list.size(); i += 8)
		{
			ranges.add(
					new FeesRange(list.get(i), list.get(i + 1), list.get(i + 2), list.get(i + 3), list.get(i + 4), list.get(i + 5), list.get(i + 6),
							FeesRange.parseList(list.get(i + 7))));
		}
		return ranges;
	}

	private static BigDecimal calcFeesFromPercentAndValue(BigDecimal value, BigDecimal feePercentage, BigDecimal feeValue, Integer decimalPlaces,
			BigDecimal maxValue)
	{
		BigDecimal totalFeesValue = BigDecimal.ZERO;
		if (ObjectChecker.isNotEmptyOrZero(feeValue))
			totalFeesValue = feeValue;
		feePercentage = ObjectChecker.toZeroIfNull(feePercentage);
		value = ObjectChecker.toZeroIfNull(value);
		totalFeesValue = totalFeesValue.add(NaMaMath.divide(value.multiply(feePercentage), new BigDecimal(100), decimalPlaces));
		if (ObjectChecker.isEmptyOrZero(maxValue))
			return totalFeesValue;
		return NaMaMath.getMin(totalFeesValue, maxValue);
	}
}
