package com.namasoft.taxauthority.ublinvoice;

import com.namasoft.common.Pair;
import com.namasoft.common.utilities.*;
import com.namasoft.taxauthority.*;
import com.namasoft.taxauthority.jordan.*;
import com.namasoft.taxauthority.ksa.entities.*;
import jakarta.xml.bind.annotation.*;

import java.math.*;
import java.text.*;
import java.util.*;

@XmlRootElement(name = "Invoice")
@XmlAccessorType(XmlAccessType.PROPERTY)
@XmlSeeAlso(value = { AbsJoFotaraInvoice.class, JoFotaraInvoice.class, JoFotaraInvoiceReturn.class, ZATCAAbsEInvoice.class, ZATCAECreditNote.class,
		ZATCAEDebitNote.class, ZATCAEInvoice.class })
public abstract class UBLAbsEInvoice extends EInvoiceDoc
{
	private String invoiceId;
	private String invoiceUUID;
	private Date issueDate;
	private Date deliveryDate;
	private String invoiceTypeCode;
	private String documentCurrencyCode;
	private String taxCurrencyCode;
	private BigDecimal taxCurrencyRate;
	private Integer invoiceLinesSize;
	private Integer invoiceCounterValue;

	private ZATCAParty issuer;
	private ZATCAParty customer;
	private String paymentMethodCode;
	private String invoiceReferenceCode;
	private String invoiceReferenceId;
	private String invoiceReferenceDescription;
	private String note;
	private String debitOrCreditNote;
	private String invoiceTransactionCode;
	private String discountReasonCode;
	private String discountReason;
	private String chargesReasonCode;
	private String chargesReason;
	private BigDecimal totalTaxAmount;
	private BigDecimal totalTaxAmountInTaxCurrency;
	private BigDecimal totalAmount;
	private BigDecimal totalNetAmount;
	private BigDecimal totalFinalAmount;
	private BigDecimal totalDiscount;
	private BigDecimal totalCharges;
	private BigDecimal invoiceHeaderCharges;
	private BigDecimal prePaidAmount;
	private BigDecimal payableRoundingAmount;
	private BigDecimal dueAmount;
	private List<ZATCAEInvoiceLine> lines;
	private List<ZATCATaxLine> totalTaxes;

	private ZATCAEGSBranchInfo branchInfo;
	private String taxpayerActivityCode;

	public UBLAbsEInvoice()
	{
		this.invoiceTypeCode = UBLEInvoiceTypeCode.TAX_INVOICE.getCode();
		this.discountReason = "Discount";
		this.discountReasonCode = "2";
		this.chargesReason = "Charges Fees";
		this.chargesReasonCode = "1";
	}

	public String getInvoiceId()
	{
		return invoiceId;
	}

	public void setInvoiceId(String invoiceId)
	{
		this.invoiceId = invoiceId;
	}

	public String getInvoiceUUID()
	{
		return invoiceUUID;
	}

	public void setInvoiceUUID(String invoiceUUID)
	{
		this.invoiceUUID = invoiceUUID;
	}

	public Date getIssueDate()
	{
		return issueDate;
	}

	public void setIssueDate(Date issueDate)
	{
		this.issueDate = issueDate;
	}

	public String getInvoiceTypeCode()
	{
		return invoiceTypeCode;
	}

	public void setInvoiceTypeCode(String invoiceTypeCode)
	{
		this.invoiceTypeCode = invoiceTypeCode;
	}

	public String getDocumentCurrencyCode()
	{
		return documentCurrencyCode;
	}

	public String fetchDocumentCurrencyCodeForXML()
	{
		return documentCurrencyCode;
	}

	public void setDocumentCurrencyCode(String documentCurrencyCode)
	{
		this.documentCurrencyCode = documentCurrencyCode;
	}

	public String getTaxCurrencyCode()
	{
		return taxCurrencyCode;
	}

	public void setTaxCurrencyCode(String taxCurrencyCode)
	{
		this.taxCurrencyCode = taxCurrencyCode;
	}

	public BigDecimal getTaxCurrencyRate()
	{
		return taxCurrencyRate;
	}

	public void setTaxCurrencyRate(BigDecimal taxCurrencyRate)
	{
		this.taxCurrencyRate = taxCurrencyRate;
	}

	public Integer getInvoiceLinesSize()
	{
		return invoiceLinesSize;
	}

	public void setInvoiceLinesSize(Integer invoiceLinesSize)
	{
		this.invoiceLinesSize = invoiceLinesSize;
	}

	public Integer getInvoiceCounterValue()
	{
		return invoiceCounterValue;
	}

	public void setInvoiceCounterValue(Integer invoiceCounterValue)
	{
		this.invoiceCounterValue = invoiceCounterValue;
	}

	public ZATCAParty getIssuer()
	{
		return issuer;
	}

	public void setIssuer(ZATCAParty issuer)
	{
		this.issuer = issuer;
	}

	public ZATCAParty getCustomer()
	{
		return customer;
	}

	public void setCustomer(ZATCAParty customer)
	{
		this.customer = customer;
	}

	public Date getDeliveryDate()
	{
		return deliveryDate;
	}

	public void setDeliveryDate(Date deliveryDate)
	{
		this.deliveryDate = deliveryDate;
	}

	public String getPaymentMethodCode()
	{
		return paymentMethodCode;
	}

	public void setPaymentMethodCode(String paymentMethodCode)
	{
		this.paymentMethodCode = paymentMethodCode;
	}

	public String getInvoiceReferenceCode()
	{
		return invoiceReferenceCode;
	}

	public void setInvoiceReferenceCode(String invoiceReferenceCode)
	{
		this.invoiceReferenceCode = invoiceReferenceCode;
	}

	public String getInvoiceReferenceId()
	{
		return invoiceReferenceId;
	}

	public void setInvoiceReferenceId(String invoiceReferenceId)
	{
		this.invoiceReferenceId = invoiceReferenceId;
	}

	public String getInvoiceReferenceDescription()
	{
		return invoiceReferenceDescription;
	}

	public void setInvoiceReferenceDescription(String invoiceReferenceDescription)
	{
		this.invoiceReferenceDescription = invoiceReferenceDescription;
	}

	public String getNote()
	{
		return note;
	}

	public void setNote(String note)
	{
		this.note = note;
	}

	public String getDebitOrCreditNote()
	{
		return debitOrCreditNote;
	}

	public void setDebitOrCreditNote(String debitOrCreditNote)
	{
		this.debitOrCreditNote = debitOrCreditNote;
	}

	public String getInvoiceTransactionCode()
	{
		return invoiceTransactionCode;
	}

	public void setInvoiceTransactionCode(String invoiceTransactionCode)
	{
		this.invoiceTransactionCode = invoiceTransactionCode;
	}

	public String getDiscountReasonCode()
	{
		return discountReasonCode;
	}

	public void setDiscountReasonCode(String discountReasonCode)
	{
		this.discountReasonCode = discountReasonCode;
	}

	public String getDiscountReason()
	{
		return discountReason;
	}

	public void setDiscountReason(String discountReason)
	{
		this.discountReason = discountReason;
	}

	public String getChargesReasonCode()
	{
		return chargesReasonCode;
	}

	public void setChargesReasonCode(String chargesReasonCode)
	{
		this.chargesReasonCode = chargesReasonCode;
	}

	public String getChargesReason()
	{
		return chargesReason;
	}

	public void setChargesReason(String chargesReason)
	{
		this.chargesReason = chargesReason;
	}

	public BigDecimal getTotalTaxAmount()
	{
		return totalTaxAmount;
	}

	public void setTotalTaxAmount(BigDecimal totalTaxAmount)
	{
		this.totalTaxAmount = totalTaxAmount;
	}

	public BigDecimal getTotalTaxAmountInTaxCurrency()
	{
		return totalTaxAmountInTaxCurrency;
	}

	public void setTotalTaxAmountInTaxCurrency(BigDecimal totalTaxAmountInTaxCurrency)
	{
		this.totalTaxAmountInTaxCurrency = totalTaxAmountInTaxCurrency;
	}

	@Override
	public BigDecimal fetchFinalTotalAmount()
	{
		return getTotalFinalAmount();
	}

	@Override
	public void updateFinalTotalAmount(BigDecimal totalAmount)
	{
		setTotalFinalAmount(totalAmount);
	}

	public BigDecimal getTotalAmount()
	{
		return totalAmount;
	}

	public void setTotalAmount(BigDecimal totalAmount)
	{
		this.totalAmount = totalAmount;
	}

	public BigDecimal getTotalNetAmount()
	{
		return totalNetAmount;
	}

	public void setTotalNetAmount(BigDecimal totalNetAmount)
	{
		this.totalNetAmount = totalNetAmount;
	}

	public BigDecimal getTotalFinalAmount()
	{
		return totalFinalAmount;
	}

	public void setTotalFinalAmount(BigDecimal totalFinalAmount)
	{
		this.totalFinalAmount = totalFinalAmount;
	}

	public BigDecimal invoiceHeaderCharges()
	{
		return NaMaMath.zeroIfNull(invoiceHeaderCharges);
	}

	@Override
	public void updateInvoiceHeaderCharges(BigDecimal invoiceHeaderCharges)
	{
		invoiceHeaderCharges(invoiceHeaderCharges);
	}

	public void invoiceHeaderCharges(BigDecimal invoiceHeaderCharges)
	{
		this.invoiceHeaderCharges = invoiceHeaderCharges;
	}

	public BigDecimal getTotalCharges()
	{
		return totalCharges;
	}

	public void setTotalCharges(BigDecimal totalCharges)
	{
		this.totalCharges = totalCharges;
	}

	public BigDecimal getTotalDiscount()
	{
		return totalDiscount;
	}

	public void setTotalDiscount(BigDecimal totalDiscount)
	{
		this.totalDiscount = totalDiscount;
	}

	public BigDecimal getPrePaidAmount()
	{
		return prePaidAmount;
	}

	public void setPrePaidAmount(BigDecimal prePaidAmount)
	{
		this.prePaidAmount = prePaidAmount;
	}

	public BigDecimal getPayableRoundingAmount()
	{
		return payableRoundingAmount;
	}

	public void setPayableRoundingAmount(BigDecimal payableRoundingAmount)
	{
		this.payableRoundingAmount = payableRoundingAmount;
	}

	public BigDecimal getDueAmount()
	{
		return dueAmount;
	}

	public void setDueAmount(BigDecimal dueAmount)
	{
		this.dueAmount = dueAmount;
	}

	public List<ZATCAEInvoiceLine> getLines()
	{
		if (lines == null)
			lines = new ArrayList<>();
		return lines;
	}

	public void setLines(List<ZATCAEInvoiceLine> lines)
	{
		this.lines = lines;
	}

	public List<ZATCATaxLine> getTotalTaxes()
	{
		if (totalTaxes == null)
			totalTaxes = new ArrayList<>();
		return totalTaxes;
	}

	public void setTotalTaxes(List<ZATCATaxLine> totalTaxes)
	{
		this.totalTaxes = totalTaxes;
	}

	public boolean simpleInvoice()
	{
		return getInvoiceTransactionCode().startsWith("02");
	}

	public String issueDateOnly()
	{
		return new SimpleDateFormat("yyyy-MM-dd").format(issueDate);
	}

	public String actualDeliveryDateOnly()
	{
		if (deliveryDate == null)
			return "";
		return new SimpleDateFormat("yyyy-MM-dd").format(deliveryDate);
	}

	public String issueTimeOnly()
	{
		if (issueDate == null)
			return "";
		return new SimpleDateFormat("hh:mm:ss").format(issueDate);
	}

	public String issueDateTime()
	{
		if (issueDate == null)
			return "";
		return new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ss").format(issueDate);
	}

	public void updateTotals()
	{
		super.updateTotals();
		setTotalTaxAmount(NaMaMath.totalizeDecimalStream(getLines().stream().map(ZATCAEInvoiceLine::itemTaxAmount)));
		BigDecimal totalAmountFromLines = NaMaMath.totalizeDecimalStream(getLines().stream().map(ZATCAEInvoiceLine::itemTotalAmount));
		BigDecimal totalDiscountsFromLines = NaMaMath.totalizeDecimalStream(getLines().stream().map(ZATCAEInvoiceLine::allowanceAmount));
		BigDecimal totalChargesFromLines = NaMaMath.totalizeDecimalStream(getLines().stream().map(ZATCAEInvoiceLine::chargeAmount));
		setTotalNetAmount(totalAmountFromLines.add(totalChargesFromLines).add(invoiceHeaderCharges()).subtract(totalDiscountsFromLines));
		updateZATCATaxes();
		setTotalTaxAmountInTaxCurrency(getTotalTaxAmount().multiply(ObjectChecker.toZeroIfNull(taxCurrencyRate)));
		setTotalDiscount(totalDiscountsFromLines);
		setTotalCharges(totalChargesFromLines.add(invoiceHeaderCharges()));
		BigDecimal amount = NaMaMath.totalizeDecimalStream(getLines().stream().map(ZATCAEInvoiceLine::itemFinalAmount));
		BigDecimal roundedAmount = NaMaMath.roundValue(amount, BigDecimal.ONE, RoundingMode.HALF_UP.name(), 0);
		if (shouldRoundPayableAmount())
			setPayableRoundingAmount(roundedAmount.subtract(amount));
		else
			setPayableRoundingAmount(BigDecimal.ZERO);
		setDueAmount(amount.add(getPayableRoundingAmount()).add(invoiceHeaderCharges()));
		setTotalFinalAmount(getTotalNetAmount().add(getTotalTaxAmount()));
		setInvoiceLinesSize(getLines().size());
	}

	public boolean shouldRoundPayableAmount()
	{
		return false;
	}

	private void updateZATCATaxes()
	{
		getTotalTaxes().clear();
		Map<Pair<String, String>, ZATCATaxLine> taxes = new HashMap<>();
		for (ZATCAEInvoiceLine line : getLines())
		{
			for (ZATCATaxLine tax : line.getTaxes())
			{
				ZATCATaxLine taxLine = taxes.computeIfAbsent(new Pair<>(tax.getTaxCategory(), tax.getTaxType()), k -> new ZATCATaxLine());
				if (!getTotalTaxes().contains(taxLine))
				{
					taxLine.setTaxPercent(tax.getTaxPercent());
					getTotalTaxes().add(taxLine);
				}
				taxLine.setTaxCategory(tax.getTaxCategory());
				taxLine.setTaxType(tax.getTaxType());
				taxLine.setTaxableAmount(NaMaMath.add(taxLine.getTaxableAmount(), line.getItemNetAmount()));
				taxLine.setTaxAmount(NaMaMath.add(taxLine.getTaxAmount(), tax.getTaxAmount()));
			}
		}
		for (ZATCATaxLine line : getTotalTaxes())
		{
			line.setTaxableAmount(line.getTaxableAmount().setScale(taxScale(), RoundingMode.HALF_UP));
			line.setTaxAmount(line.getTaxAmount().setScale(taxScale(), RoundingMode.HALF_UP));
		}
		setTotalTaxAmount(NaMaMath.totalizeDecimalStream(getTotalTaxes().stream().map(ZATCATaxLine::getTaxAmount)));
	}

	@Override
	public void roundValues(int scale)
	{
		getLines().forEach(l -> l.roundAmounts(scale));
		setTotalTaxAmount(NaMaMath.round(getTotalTaxAmount(), scale).stripTrailingZeros());
		setTotalTaxAmountInTaxCurrency(NaMaMath.round(getTotalTaxAmountInTaxCurrency(), scale).stripTrailingZeros());
		setTotalNetAmount(NaMaMath.round(getTotalNetAmount(), scale).stripTrailingZeros());
		setTotalFinalAmount(NaMaMath.round(getTotalFinalAmount(), scale).stripTrailingZeros());
		setTotalDiscount(NaMaMath.round(getTotalDiscount(), scale).stripTrailingZeros());
		setTotalCharges(NaMaMath.round(getTotalCharges(), scale).stripTrailingZeros());
		setDueAmount(NaMaMath.round(getDueAmount(), scale).stripTrailingZeros());
	}

	public int linesCount()
	{
		return getLines().size();
	}

	@Override
	public void updateDocumentVersion(boolean signed)
	{

	}

	@Override
	public void updateTaxpayerActivityCode(String activityCode)
	{
		this.taxpayerActivityCode = activityCode;
		branchInfo().setBranchIndustry(activityCode);
	}

	public String getTaxpayerActivityCode()
	{
		return taxpayerActivityCode;
	}

	public void setTaxpayerActivityCode(String taxpayerActivityCode)
	{
		this.taxpayerActivityCode = taxpayerActivityCode;
	}

	public ZATCAEGSBranchInfo branchInfo()
	{
		if (this.branchInfo != null)
			return this.branchInfo;
		return this.branchInfo = new ZATCAEGSBranchInfo();
	}

	private ZATCAParty issuer()
	{
		if (this.issuer != null)
			return this.issuer;
		return this.issuer = new ZATCAParty();
	}

	protected ZATCAParty customer()
	{
		if (this.customer != null)
			return this.customer;
		return this.customer = new ZATCAParty();
	}

	@Override
	public void updateDateTimeIssued(String dateTimeIssued)
	{
		try
		{
			setIssueDate(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss").parse(dateTimeIssued));
		}
		catch (ParseException e)
		{

		}
	}

	@Override
	public String fetchDateTimeIssued()
	{
		return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss").format(this.issueDate);
	}

	@Override
	public void updateIssuer(TaxAuthorityEntityWithAddress issuer)
	{
		copyIssuerData(issuer(), issuer);
	}

	private static void copyIssuerData(ZATCAParty issuerParty, TaxAuthorityEntityWithAddress issuer)
	{
		TaxAuthorityAddress address = issuer.address();
		issuerParty.setStreetName(address.getStreet());
		issuerParty.setBuildingNumber(address.getBuildingNumber());
		issuerParty.setCityName(address.getGovernate());
		issuerParty.setCitySubdivisionName(address.getDistrict());
		issuerParty.setCountrySubEntity(address.getRegionCity());
		issuerParty.setPostalZone(address.getPostalCode());
		issuerParty.setCountryCode(address.getCountry());
		issuerParty.setEntityTaxType(issuer.getType());
		issuerParty.setIdentityType("CRN");
		issuerParty.setPlotIdentification(address.getLandmark());
		issuerParty.setIdentification(address.fetchBranchId());
		issuerParty.setVatRegistrationNumber(issuer.getId());
		issuerParty.setVatRegistrationName(issuer.getName());
	}

	public static ZATCAParty createIssuerFrom(TaxAuthorityIssuer issuer)
	{
		ZATCAParty party = new ZATCAParty();
		copyIssuerData(party, issuer);
		return party;
	}

	@Override
	public TaxAuthorityIssuer fetchIssuer()
	{
		TaxAuthorityIssuer issuer = new TaxAuthorityIssuer();
		TaxAuthorityIssuerAddress address = new TaxAuthorityIssuerAddress();
		issuer.setAddress(address);
		address.setStreet(issuer().getStreetName());
		address.setBuildingNumber(issuer().getBuildingNumber());
		address.setGovernate(issuer().getCityName());
		address.setDistrict(issuer().getCitySubdivisionName());
		address.setRegionCity(issuer().getCountrySubEntity());
		address.setPostalCode(issuer().getPostalZone());
		address.setCountry(issuer().getCountryCode());
		issuer.setType(issuer().getEntityTaxType());
		address.setBranchId(issuer().getIdentification());
		issuer.setId(issuer().getVatRegistrationNumber());
		issuer.setName(issuer().getVatRegistrationName());
		return issuer;
	}

	@Override
	public void updateDelivery(TaxAuthorityDelivery delivery)
	{

	}

	@Override
	public TaxAuthorityDelivery fetchDelivery()
	{
		return null;
	}

	@Override
	public void updateReceiver(TaxAuthorityReceiver receiver)
	{
		if (receiver == null)
			return;
		TaxAuthorityAddress address = receiver.getAddress();
		customer().setStreetName(address.getStreet());
		customer().setPlotIdentification(address.getLandmark());
		customer().setBuildingNumber(address.getBuildingNumber());
		customer().setCityName(address.getGovernate());
		customer().setCitySubdivisionName(address.getDistrict());
		customer().setCountrySubEntity(address.getRegionCity());
		customer().setPostalZone(address.getPostalCode());
		customer().setCountryCode(address.getCountry());
		customer().setEntityTaxType(receiver.getType());
		updateReceiverIdentityType(receiver);
		customer().setIdentification(receiver.getId());
		customer().setVatRegistrationName(receiver.getName());
	}

	public abstract void updateReceiverIdentityType(TaxAuthorityReceiver receiver);

	@Override
	public TaxAuthorityReceiver fetchReceiver()
	{
		TaxAuthorityReceiver receiver = new TaxAuthorityReceiver();
		TaxAuthorityAddress address = receiver.getAddress();
		address.setStreet(customer().getStreetName());
		address.setBuildingNumber(customer().getBuildingNumber());
		address.setGovernate(customer().getCityName());
		address.setDistrict(customer().getCitySubdivisionName());
		address.setRegionCity(customer().getCountrySubEntity());
		address.setPostalCode(customer().getPostalZone());
		address.setCountry(customer().getCountryCode());
		receiver.setType(customer().getEntityTaxType());
		receiver.setId(customer().getIdentification());
		return receiver;
	}

	@Override
	public void updatePayment(TaxAuthorityPayment payment)
	{

	}

	@Override
	public TaxAuthorityPayment fetchPayment()
	{
		return null;
	}

	@Override
	public void updateTotalSalesAmount(BigDecimal totalSalesAmount)
	{
		setTotalAmount(totalSalesAmount);
	}

	@Override
	public void updateTotalDiscountAmount(BigDecimal totalDiscountAmount)
	{
		setTotalDiscount(totalDiscountAmount);
	}

	@Override
	public BigDecimal fetchTotalDiscountAmount()
	{
		return getTotalDiscount();
	}

	@Override
	public void updateNetAmount(BigDecimal netAmount)
	{
		setTotalFinalAmount(netAmount);
	}

	@Override
	public void updateExtraDiscountAmount(BigDecimal extraDiscountAmount)
	{

	}

	@Override
	public void updateTotalItemsDiscountAmount(BigDecimal totalItemsDiscountAmount)
	{

	}

	@Override
	public BigDecimal fetchTotalItemsDiscountAmount()
	{
		return BigDecimal.ZERO;
	}

	@Override
	public void updateInternalCode(String internalCode)
	{
		setInvoiceId(internalCode);
	}

	@Override
	public String fetchInternalCode()
	{
		return getInvoiceId();
	}

	@Override
	public void addLine(EInvoiceDocLine taxInvoiceLine)
	{
		getLines().add((ZATCAEInvoiceLine) taxInvoiceLine);
	}

	@Override
	public boolean documentVersionRequireSignature()
	{
		return false;
	}

	@Override
	public boolean shouldValidateSignature()
	{
		return false;
	}

	@Override
	public List<TaxAuthoritySignature> getSignatures()
	{
		return null;
	}

	@Override
	public void setSignatures(List<TaxAuthoritySignature> signatures)
	{

	}

	@Override
	public void updateLines(List<? extends EInvoiceDocLine> lines)
	{
		setLines(CollectionsUtility.castList(lines));
	}

	@Override
	public List<EInvoiceDocLine> fetchLines()
	{
		return CollectionsUtility.castList(getLines());
	}

	@Override
	public BigDecimal fetchTotalSalesAmount()
	{
		return getTotalAmount();
	}

	@Override
	public BigDecimal fetchNetAmount()
	{
		return getTotalNetAmount();
	}

	@Override
	public void updateUUID(String uuid)
	{
		setInvoiceUUID(uuid);
	}

	@Override
	public void updateDocumentCurrency(String code)
	{
		setDocumentCurrencyCode(code);
	}

	@Override
	public void updateTaxCurrencyCode(String code)
	{
		setTaxCurrencyCode(code);
	}

	@Override
	public boolean receiverIdRequired(BigDecimal min)
	{
		return ObjectChecker.isFalse(simpleInvoice());
	}

	@Override
	public boolean receiverNameIsRequired(BigDecimal min)
	{
		return ObjectChecker.isFalse(simpleInvoice());
	}

	@Override
	public boolean shouldCheckReceiverAddress()
	{
		return false;
	}

	@Override
	public void updateHeaderCurrencyRate(BigDecimal rate)
	{
		setTaxCurrencyRate(rate);
	}

	@Override
	public <T extends EInvoiceDocLine> T createLine()
	{
		return (T) new ZATCAEInvoiceLine();
	}

	@Override
	public int taxScale()
	{
		return 9;
	}

	@Override
	public String fetchHeaderCurrency()
	{
		return getDocumentCurrencyCode();
	}

	@Override
	public BigDecimal fetchHeaderCurrencyRate()
	{
		return getTaxCurrencyRate();
	}

	@Override
	public String fetchDebitOrCreditNote()
	{
		return getDebitOrCreditNote();
	}

	@Override
	public String fetchPaymentMethod()
	{
		return getPaymentMethodCode();
	}

	@Override
	public void updateDebitOrCreditNote(String debitOrCreditNote)
	{
		if (invoice())
			return;
		setDebitOrCreditNote(debitOrCreditNote);
	}

	public boolean invoice()
	{
		return false;
	}

	@Override
	public void updatePaymentMethod(String paymentMethodCode)
	{
		setPaymentMethodCode(paymentMethodCode);
	}

	@Override
	public void updateSalesOrderReference(String salesOrderReference)
	{
		setInvoiceReferenceCode(salesOrderReference);
	}

	public int updateICVAndGetNewValue(int lastCounterValue)
	{
		setInvoiceCounterValue(lastCounterValue = lastCounterValue + 1);
		return lastCounterValue;
	}

	@Override
	public boolean hasDefaultTaxCurrency()
	{
		return true;
	}

	@Override
	public String cashPaymentMethodCode()
	{
		return "10";
	}

	@Override
	public String otherPaymentMethodCode()
	{
		return "1";
	}

	public String fetchPreviousInvoiceHash()
	{
		return null;
	}

	public boolean zatcaPortal()
	{
		return false;
	}

	public boolean joFotaraPortal()
	{
		return false;
	}

	@Override
	public void updateReferenceUUID(String uuid)
	{
		setInvoiceReferenceId(uuid);
	}

	@Override
	public void updateSalesOrderDescription(String salesOrderDescription)
	{
		setInvoiceReferenceDescription(salesOrderDescription);
	}

	@Override
	public String fetchSalesOrderDescription()
	{
		return getInvoiceReferenceDescription();
	}

	@Override
	public boolean ublInvoice()
	{
		return true;
	}

	public String fetchExemptionCode()
	{
		return "E";
	}

	public BigDecimal calculateLineExtensionAmount()
	{
		return NaMaMath.totalizeDecimalStream(getLines().stream().map(ZATCAEInvoiceLine::itemTotalAmount));
	}

	public BigDecimal calculateTaxExclusiveAmount()
	{
		return calculateLineExtensionAmount();
	}

	public BigDecimal calculatePayableAmount()
	{
		return getTotalFinalAmount().add(getPayableRoundingAmount());
	}

	public BigDecimal calculateTaxInclusiveAmount()
	{
		return calculateTaxExclusiveAmount().add(getTotalTaxAmount()).subtract(getTotalDiscount());
	}
}
