package com.namasoft.common.utils.importer.data;

import com.namasoft.common.NaMaDTO;
import com.namasoft.common.fieldids.CommonFieldIds;
import com.namasoft.common.flatobjects.NaMaError;
import com.namasoft.common.layout.edit.NaMaText;
import com.namasoft.common.utilities.*;
import com.namasoft.common.utils.importer.parser.ImportDataManager;

import jakarta.xml.bind.annotation.*;
import java.util.*;
import java.util.function.Function;

@XmlRootElement
@XmlAccessorType(XmlAccessType.PROPERTY)
public class DataRecord extends NaMaDTO
{
	private String entity;
	private List<FieldValue> fields = new ArrayList<>();
	private List<DetailEntry> details = new ArrayList<>();
	private Boolean inserted;
	private Integer lineNumber;
	private ImportDataManager source;
	private String sheet;
	private String finderField;
	private String finderValue;
	private String uniqueBy;
	private Boolean saveAsDraft;
	private Boolean doNotAddRecords;
	private Boolean doNotUpdateRecords;
	private List<String> errors;

	public DataRecord(ImportDataManager input)
	{
		this();
		this.source = input;
		if (input != null)
		{
			this.lineNumber = input.currentLineNumber();
			this.sheet = input.currentSheet();
		}
	}

	public DataRecord()
	{
	}

	public String getEntity()
	{
		return entity;
	}

	public void setEntity(String entity)
	{
		this.entity = entity;
	}

	public List<FieldValue> getFields()
	{
		return fields;
	}

	public void setFields(List<FieldValue> fields)
	{
		this.fields = fields;
	}

	public List<DetailEntry> getDetails()
	{
		return details;
	}

	public void setDetails(List<DetailEntry> details)
	{
		this.details = details;
	}

	public Boolean getInserted()
	{
		return inserted;
	}

	public void setInserted(Boolean inserted)
	{
		this.inserted = inserted;
	}

	@Override
	public String toString()
	{
		StringBuilder toStr = new StringBuilder();
		toStr.append(getEntity()).append("{");
		for (FieldValue entry : getFields())
		{
			toStr.append(entry).append(",");
		}
		for (DetailEntry entry : getDetails())
		{
			toStr.append(entry).append(",");
		}
		toStr.append("}\n");
		return toStr.toString();
	}

	public Integer getLineNumber()
	{
		return lineNumber;
	}

	public String getSheet()
	{
		return sheet;
	}

	public void setLineNumber(Integer lineNumber)
	{
		this.lineNumber = lineNumber;
	}

	@XmlTransient
	public ImportDataManager getSource()
	{
		return source;
	}

	public void setSource(ImportDataManager source)
	{
		this.source = source;
	}

	public String getFinderField()
	{
		return finderField;
	}

	public void setFinderField(String finderField)
	{
		this.finderField = finderField;
	}

	public String getFinderValue()
	{
		return finderValue;
	}

	public void setFinderValue(String finderValue)
	{
		this.finderValue = finderValue;
	}

	public Boolean getSaveAsDraft()
	{
		return saveAsDraft;
	}

	public void setSaveAsDraft(Boolean saveAsDraft)
	{
		this.saveAsDraft = saveAsDraft;
	}

	public Boolean getDoNotAddRecords()
	{
		return doNotAddRecords;
	}

	public void setDoNotAddRecords(Boolean doNotAddRecords)
	{
		this.doNotAddRecords = doNotAddRecords;
	}

	public Boolean getDoNotUpdateRecords()
	{
		return doNotUpdateRecords;
	}

	public void setDoNotUpdateRecords(Boolean doNotUpdateRecords)
	{
		this.doNotUpdateRecords = doNotUpdateRecords;
	}

	public List<String> getErrors()
	{
		return errors;
	}

	public void setErrors(List<String> errors)
	{
		this.errors = errors;
	}

	public void markAsInserted()
	{
		if (getSource() != null)
			getSource().markRecordAsInserted(this);
		setInserted(true);
	}

	public void markAsError(List<NaMaError> errors, Function<NaMaText, String> translator)
	{
		DataRecord thisRecord = this;
		if (getSource() != null)
			getSource().markRecordAsError(thisRecord, errors);
		else
			this.addErrors(errors, translator);
		List<DetailEntry> details = getDetails();
		if (ObjectChecker.isEmptyOrNull(details))
			return;
		errors.stream().filter(e -> e.getRow() != null && e.getRow() > -1 && e.getOwnerId() != null).forEach(e -> {
			DetailEntry entry = details.stream().filter(d -> e.getOwnerId().startsWith(d.getTable() + ".") || e.getOwnerId().equals(d.getTable()))
					.findFirst().orElse(null);
			if (entry == null || ObjectChecker.isEmptyOrNull(entry.getRows()))
				return;
			if (entry.getRows().size() <= e.getRow())
				return;
			if (getSource() != null)
				getSource().markDetailLineAsError(thisRecord, entry.getId(), entry.getRows().get(e.getRow()), e);
			else
				entry.getRows().get(e.getRow()).addError(NaMaError.describeErrors(Collections.singletonList(e), "\n", translator));
		});
	}

	private void addErrors(List<NaMaError> errors, Function<NaMaText, String> translator)
	{
		if (getErrors() == null)
			setErrors(new ArrayList<>());
		getErrors().add(NaMaError.describeErrors(errors, "\n", translator));
	}

	public boolean ifRecordCodeAndEntityTypeEqual(String code, String entityType)
	{
		boolean codeMatched = getFields().stream().filter(ObjectChecker.equalsPredicate("code", FieldValue::getId))
				.anyMatch(ObjectChecker.equalsPredicate(code, FieldValue::getValue));
		boolean entityTypeMatched = ObjectChecker.areEqual(entityType, getEntity());
		return codeMatched && entityTypeMatched;
	}

	public String code()
	{
		return fetchFieldValue("code");
	}

	public String altCode()
	{
		return fetchFieldValue("altCode");
	}

	public String id()
	{
		return fetchFieldValue("id");
	}

	public void updateErrorsInfo(List<NaMaError> errors, Integer recordIndex)
	{
		CollectionsUtility.setInDetails(errors, NaMaError::setRecordCode, code());
		CollectionsUtility.setInDetails(errors, NaMaError::setRecordIndexInRequest, recordIndex);
	}

	public String getUniqueBy()
	{
		return uniqueBy;
	}

	public void setUniqueBy(String uniqueBy)
	{
		this.uniqueBy = uniqueBy;
	}

	public void mergeInto(DataRecord other)
	{
		for (FieldValue entry : getFields())
		{
			FieldValue otherEntry = other.getFields().stream().filter(ObjectChecker.equalsPredicate(entry.getId(), FieldValue::getId)).findFirst()
					.orElse(null);
			if (otherEntry == null)
				other.getFields().add(entry);
			else
				otherEntry.mergeWith(entry);
		}
		for (DetailEntry entry : getDetails())
		{
			DetailEntry otherEntry = other.getDetails().stream().filter(ObjectChecker.equalsPredicate(entry.getId(), DetailEntry::getId)).findFirst()
					.orElse(null);
			if (otherEntry == null)
				other.getDetails().add(entry);
			else
				otherEntry.mergeWith(entry);
		}
	}

	public String fetchFieldValue(String fieldId)
	{
		return getFields().stream().filter(ObjectChecker.equalsPredicate(fieldId, FieldValue::getId)).map(FieldValue::getValue).findFirst()
				.orElse(null);
	}

	public DetailEntry fetchDetail(String detailField)
	{
		return getDetails().stream().filter(ObjectChecker.equalsPredicate(detailField, DetailEntry::getId)).findFirst().orElse(null);
	}

	public boolean sameAs(DataRecord other)
	{
		if (other == null)
			return false;
		if (ObjectChecker.areNotEqual(getEntity(), other.getEntity()))
			return false;
		if (ObjectChecker.isAnyNotEmptyOrNull(other.getUniqueBy(), getUniqueBy()))
			return ObjectChecker.areEqual(other.getUniqueBy(), getUniqueBy());
		String myCode = fetchFieldValue(CommonFieldIds.CODE);
		String otherCode = other.fetchFieldValue(CommonFieldIds.CODE);
		return ObjectChecker.areAllNotEmptyOrNull(myCode, otherCode) && ObjectChecker.areEqual(myCode, otherCode);
	}

	public boolean areFieldsAndDetailsEmpty()
	{
		return !(anyFieldHasValue() || anyDetailHasValues());
	}

	public boolean anyDetailHasValues()
	{
		return getDetails().stream().map(DetailEntry::getRows).flatMap(Collection::stream).anyMatch(DetailEntryRow::anyColumnHasValue);
	}

	public boolean anyFieldHasValue()
	{
		return getFields().stream().filter(Objects::nonNull).map(FieldValue::getValue).anyMatch(ObjectChecker::isNotEmptyOrNull);
	}
}
