package com.namasoft.taxauthority;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.namasoft.common.utilities.*;

import jakarta.xml.bind.annotation.*;
import java.math.BigDecimal;
import java.util.*;

@XmlRootElement
@XmlAccessorType(XmlAccessType.PROPERTY)
public class TaxAuthorityInvoiceLine extends EInvoiceDocLine
{
	private String description;
	private String itemType;
	private String itemCode;
	private String unitType;
	private String internalCode;
	private BigDecimal quantity;
	private BigDecimal salesTotal;
	private BigDecimal total;
	private BigDecimal valueDifference = BigDecimal.ZERO;
	private BigDecimal totalTaxableFees;
	private BigDecimal netTotal;
	private BigDecimal itemsDiscount;
	private TaxAuthorityUnitValue unitValue;
	private TaxAuthorityDiscount discount;
	private List<TaxAuthorityLineTax> taxableItems;

	public TaxAuthorityInvoiceLine()
	{
		this.taxableItems = new ArrayList<>();
	}

	public String getDescription()
	{
		return description;
	}

	public void setDescription(String description)
	{
		this.description = description;
	}

	public String getItemType()
	{
		return itemType;
	}

	public void setItemType(String itemType)
	{
		this.itemType = itemType;
	}

	public String getItemCode()
	{
		return itemCode;
	}

	public void setItemCode(String itemCode)
	{
		this.itemCode = itemCode;
	}

	public String getUnitType()
	{
		return unitType;
	}

	public void setUnitType(String unitType)
	{
		this.unitType = unitType;
	}

	@Override
	public void updateDiscount(TaxAuthorityDiscount discount)
	{
		setDiscount(discount);
	}

	@Override
	public TaxAuthorityDiscount fetchDiscount()
	{
		return getDiscount();
	}

	@Override
	public void updateTaxableItems(List<TaxAuthorityLineTax> taxableItems)
	{
		setTaxableItems(taxableItems);
	}

	@Override
	public List<TaxAuthorityLineTax> fetchTaxableItems()
	{
		return getTaxableItems();
	}

	@Override
	public void updateTotalTaxableFees(BigDecimal totalTaxableFees)
	{
		setTotalTaxableFees(totalTaxableFees);
	}

	@Override
	public void updateItemsDiscount(BigDecimal itemsDiscount)
	{
		setItemsDiscount(itemsDiscount);
	}

	@Override
	public void updateItemCode(String itemCode)
	{
		setItemCode(itemCode);
	}

	@Override
	public String fetchItemCode()
	{
		return getItemCode();
	}

	@Override
	public void updateItemType(String itemType)
	{
		setItemType(itemType);
	}

	@Override
	public void updateInternalCode(String internalCode)
	{
		setInternalCode(internalCode);
	}

	@Override
	public void updateDescription(String description)
	{
		setDescription(description);
	}

	public String getInternalCode()
	{
		return internalCode;
	}

	public void setInternalCode(String internalCode)
	{
		this.internalCode = internalCode;
	}

	public BigDecimal getQuantity()
	{
		return quantity;
	}

	public void setQuantity(BigDecimal quantity)
	{
		this.quantity = quantity;
	}

	public BigDecimal getSalesTotal()
	{
		return salesTotal;
	}

	public void setSalesTotal(BigDecimal salesTotal)
	{
		this.salesTotal = salesTotal;
	}

	public BigDecimal getTotal()
	{
		return total;
	}

	public void setTotal(BigDecimal total)
	{
		this.total = total;
	}

	public BigDecimal getValueDifference()
	{
		return valueDifference;
	}

	public void setValueDifference(BigDecimal valueDifference)
	{
		this.valueDifference = valueDifference;
	}

	public BigDecimal getTotalTaxableFees()
	{
		return totalTaxableFees;
	}

	public void setTotalTaxableFees(BigDecimal totalTaxableFees)
	{
		this.totalTaxableFees = totalTaxableFees;
	}

	public BigDecimal getNetTotal()
	{
		return netTotal;
	}

	public void setNetTotal(BigDecimal netTotal)
	{
		this.netTotal = netTotal;
	}

	public BigDecimal getItemsDiscount()
	{
		return itemsDiscount;
	}

	public void setItemsDiscount(BigDecimal itemsDiscount)
	{
		this.itemsDiscount = itemsDiscount;
	}

	public TaxAuthorityUnitValue getUnitValue()
	{
		return unitValue;
	}

	public void setUnitValue(TaxAuthorityUnitValue unitValue)
	{
		this.unitValue = unitValue;
	}

	public TaxAuthorityDiscount getDiscount()
	{
		return discount;
	}

	public void setDiscount(TaxAuthorityDiscount discount)
	{
		this.discount = discount;
	}

	public List<TaxAuthorityLineTax> getTaxableItems()
	{
		return taxableItems;
	}

	public void setTaxableItems(List<TaxAuthorityLineTax> taxableItems)
	{
		this.taxableItems = taxableItems;
	}

	@JsonIgnore
	public void updateNulls()
	{
		this.unitValue = ObjectChecker.getFirstNotNullObj(getUnitValue(), new TaxAuthorityUnitValue());
		this.discount = ObjectChecker.getFirstNotNullObj(getDiscount(), new TaxAuthorityDiscount());
		this.taxableItems = ObjectChecker.getFirstNotNullObj(getTaxableItems(), new ArrayList<>());
	}

	public void calcNetTotal()
	{
		setNetTotal(NaMaMath.zeroIfNull(getSalesTotal()).subtract(NaMaMath.zeroIfNull(getDiscount().getAmount())));
	}

	@Override
	public void updateNetTotal(BigDecimal netTotal)
	{
		setNetTotal(netTotal);
	}

	@Override
	public BigDecimal fetchNetTotal()
	{
		return getNetTotal();
	}

	public void calcTotal()
	{
		updateItemsDiscountForNonNegativeValues();
		setTotal(totalWithoutDiscount().subtract(NaMaMath.zeroIfNull(itemsDiscount)));
	}

	@Override
	public void updateTotal(BigDecimal total)
	{
		setTotal(total);
	}

	@Override
	public BigDecimal fetchFinalTotal()
	{
		return getTotal();
	}

	private BigDecimal totalWithoutDiscount()
	{
		BigDecimal totalTaxAmount = NaMaMath.totalizeDecimalStream(getTaxableItems().stream()
				.map(l -> ObjectChecker.isTrue(l.getDiscount()) ? NaMaMath.multiply(l.getAmount(), BigDecimal.valueOf(-1)) : l.getAmount()));
		return NaMaMath.sum(netTotal, totalTaxAmount, valueDifference);
	}

	private void updateItemsDiscountForNonNegativeValues()
	{
		BigDecimal totalWithoutDiscount = totalWithoutDiscount();
		if (totalWithoutDiscount.subtract(itemsDiscount).compareTo(BigDecimal.ZERO) < 0)
			itemsDiscount = totalWithoutDiscount;
	}

	@Override
	public void calcSalesTotal()
	{
		setSalesTotal(NaMaMath.multiply(getQuantity(), getUnitValue().getAmountEGP()));
	}

	@Override
	public void updateSalesTotal(BigDecimal salesTotal)
	{
		setSalesTotal(salesTotal);
	}

	@Override
	public BigDecimal fetchSalesTotal()
	{
		return getSalesTotal();
	}

	@Override
	public void updateUnitPrice(TaxAuthorityUnitValue unitValue)
	{
		setUnitValue(unitValue);
	}

	@Override
	public BigDecimal fetchItemsDiscount()
	{
		return getItemsDiscount();
	}

	@Override
	public void updateUnitType(String unitType)
	{
		setUnitType(unitType);
	}

	@Override
	public String fetchUnitType()
	{
		return getUnitType();
	}

	@Override
	public String currencySold(EInvoiceDoc doc)
	{
		return getUnitValue().getCurrencySold();
	}

	@Override
	public String fetchItemType()
	{
		return getItemType();
	}

	@Override
	public String fetchDescription()
	{
		return getDescription();
	}

	@Override
	public String fetchInternalCode()
	{
		return getInternalCode();
	}

	@Override
	public BigDecimal fetchUnitPrice()
	{
		return getUnitValue().getAmountEGP();
	}

	@Override
	public BigDecimal currencyRate(EInvoiceDoc doc)
	{
		return getUnitValue().getCurrencyExchangeRate();
	}
}